//=============================================================================
// Vitsuno_BattleStyle.js
//=============================================================================
// [更新情報]
// Ver 1.24 : カーソル状態微修正、TPの説明を追加。
// Ver 1.23 : 名前描画機能追加。
// Ver 1.22 : 不具合修正。
// Ver 1.21 : カーソル選択とエネミー選択対象を微修正。
// Ver 1.20 : 互換性のため、カーソル選択の実装を変更。
// Ver 1.13 : ステータスウィンドウの互換性を向上。
// Ver 1.12 : カーソル選択の表示範囲の定義を変更。
// Ver 1.11 : アクターコマンドを画面内に収めるように変更。
//          : ステータスウィンドウを他のウィンドウより下に表示。
// Ver 1.10 : アクターコマンド表示・非表示処理を変更！
// Ver 1.00 : 基本版完成！
//-----------------------------------------------------------------------------
// Copyright (c) 2018 Tsuno Ozumi
// Released under the MIT license
// https://opensource.org/licenses/mit-license.php
//-----------------------------------------------------------------------------
// Site : https://vitsuno.net/
//=============================================================================

/*:
 * @plugindesc ＜バトル＞バトル画面のレイアウトを変更します。
 * @author 尾角 つの (Tsuno Ozumi)
 *
 * @param (Target Settings)
 * @desc このメモは、ご自由にお使いください。
 * @type note
 * @default ""
 *
 * @param Target Cursor Image
 * @desc ターゲット選択画像
 * @require 1
 * @dir img/system/
 * @type file
 * @default BattleTargetCursor
 *
 * @param
 *
 * @param (Status Settings)
 * @desc このメモは、ご自由にお使いください。
 * @type note
 * @default ""
 *
 * @param Status Skin Image
 * @desc ステータススキン画像
 * @require 1
 * @dir img/system/
 * @type file
 * @default BattleStatusSkin
 *
 * @param Status Cursor Image
 * @desc ステータス選択画像
 * @require 1
 * @dir img/system/
 * @type file
 *
 * @param Actor Skin Settings
 * @desc アクタースキン設定
 * @type struct<actorSkin>
 * @default {"Graphic Type":"none","Graphic X":"28","Graphic Y":"8","Name Size":"16","Name X":"8","Name Y":"110","Name Width":"184","Name Align":"left"}
 *
 * @param HP Skin Settings
 * @desc HPスキン設定
 * @type struct<paramSkin>
 * @default {"Number Image":"BattleLargeNumber","Number X":"156","Number Y":"138","Gauge Image":"BattleHpGauge","Gauge X":"8","Gauge Y":"166"}
 *
 * @param MP Skin Settings
 * @desc MPスキン設定
 * @type struct<paramSkin>
 * @default {"Number Image":"BattleSmallNumber","Number X":"98","Number Y":"174","Gauge Image":"BattleMpGauge","Gauge X":"8","Gauge Y":"194"}
 *
 * @param TP Skin Settings
 * @desc TPスキン設定
 * @type struct<paramSkin>
 * @default {"Number Image":"BattleSmallNumber","Number X":"192","Number Y":"174","Gauge Image":"BattleTpGauge","Gauge X":"102","Gauge Y":"194"}
 *
 * @param State Skin Settings
 * @desc ステートアイコン設定
 * @type struct<stateSkin>
 * @default {"Icon X":"176","Icon Y":"154"}
 *
 * @param
 *
 * @param (Side View Settings)
 * @desc このメモは、ご自由にお使いください。
 * @type note
 * @default ""
 *
 * @param Adjust Position X
 * @desc サイドビューのアクター位置のX座標の調整
 * @type number
 * @max 9999
 * @min -9999
 * @default 0
 *
 * @param Adjust Position Y
 * @desc サイドビューのアクター位置のY座標の調整
 * @type number
 * @max 9999
 * @min -9999
 * @default -30
 * 
 * @help
 * 
 * バトル中の画面のレイアウトを変更するプラグインです。
 * スキンを使用したステータス表示や、カーソルを用いたターゲット選択が行えます。
 *
 * ＜必須プラグイン＞
 * 　以下のプラグインを、このプラグインより上に導入して下さい。
 *
 * 　Vitsuno_SpritePerch.js    // 画像表示機能プラグイン
 *
 * ＜画像設定＞
 * 　以下の画像を用意して、パラメータから設定してください。
 *
 * 　ターゲット選択画像　（必須）
 * 　ステータススキン画像　（必須）
 * 　ステータス選択画像
 * 　HP数字画像
 * 　HPゲージ画像
 * 　MP数字画像
 * 　MPゲージ画像
 * 　TP数字画像
 * 　TPゲージ画像
 *
 * ＜詳細と注意点＞
 * ・TPは、システムのオプションで非表示にしていても表示されます。
 * ・数字やゲージを表示しない場合は、画像を（なし）に設定してください。
 * ・ステータススキン画像の大きさが、ステータス配置の基準となります。
 */

/*~struct~actorSkin:
 *
 * @param Graphic Type
 * @desc グラフィックの種類
 * @type select
 * @option 顔グラフィック
 * @value face
 * @option なし
 * @value none
 *
 * @param Graphic X
 * @desc グラフィックのX座標
 * @type number
 *
 * @param Graphic Y
 * @desc グラフィックのY座標
 * @type number
 *
 * @param Name Size
 * @desc 名前の大きさ（使用しない場合は0）
 * @type number
 *
 * @param Name X
 * @desc 名前のX座標
 * @type number
 *
 * @param Name Y
 * @desc 名前のY座標
 * @type number
 *
 * @param Name Width
 * @desc 名前の最大幅
 * @type number
 *
 * @param Name Align
 * @desc 名前の配置
 * @type select
 * @option 左寄り
 * @value left
 * @option 中央寄り
 * @value center
 * @option 右寄り
 * @value right
 *
 */

/*~struct~paramSkin:
 *
 * @param Number Image
 * @desc 数値画像
 * @require 1
 * @dir img/system/
 * @type file
 *
 * @param Number X
 * @desc 数値のX座標 (右端)
 * @type number
 *
 * @param Number Y
 * @desc 数値のY座標
 * @type number
 *
 * @param Gauge Image
 * @desc ゲージ画像
 * @require 1
 * @dir img/system/
 * @type file
 *
 * @param Gauge X
 * @desc ゲージのX座標
 * @type number
 * 
 * @param Gauge Y
 * @desc ゲージのY座標
 * @type number
 *
 */

/*~struct~stateSkin:
 *
 * @param Icon X
 * @desc アイコンのX座標
 * @type number
 *
 * @param Icon Y
 * @desc アイコンのY座標
 * @type number
 *
 */

var Vitsuno = Vitsuno || {};

if (!Vitsuno.PERCH_VERSION) {
    console.error('Vitsuno_BattleStyle.js : Vitsuno_SpritePerch.js is required.');
}

(function ($) {
    'use strict';

    // バージョン情報
    $.BATTLE_STYLE_VERSION = 1.24;

    // プラグインパラメータの処理
    var _getActorSkin = function (json) {
        var data = JSON.parse(json);
        var skin = {};
        skin.graphicType = data['Graphic Type'] || 'none';
        skin.graphicX = Number(data['Graphic X'] || 0);
        skin.graphicY = Number(data['Graphic Y'] || 0);
        skin.nameSize = Number(data['Name Size'] || 0);
        skin.nameX = Number(data['Name X'] || 0);
        skin.nameY = Number(data['Name Y'] || 0);
        skin.nameWidth = Number(data['Name Width'] || 0);
        skin.nameAlign = data['Name Align'] || 'left';
        return skin;
    };

    var _getParamSkin = function (json) {
        var data = JSON.parse(json);
        var skin = {};
        skin.numberImage = data['Number Image'] || '';
        skin.numberX = Number(data['Number X'] || 0);
        skin.numberY = Number(data['Number Y'] || 0);
        skin.gaugeImage = data['Gauge Image'] || '';
        skin.gaugeX = Number(data['Gauge X'] || 0);
        skin.gaugeY = Number(data['Gauge Y'] || 0);
        return skin;
    };

    var _getStateSkin = function (json) {
        var data = JSON.parse(json);
        var skin = {};
        skin.iconX = Number(data['Icon X'] || 0);
        skin.iconY = Number(data['Icon Y'] || 0);
        return skin;
    };

    // プラグインの設定値を取得
    var _plugin = PluginManager.parameters('Vitsuno_BattleStyle');

    var _actorSkin = _getActorSkin(_plugin['Actor Skin Settings'] || '{}');
    var _hpSkin = _getParamSkin(_plugin['HP Skin Settings'] || '{}');
    var _mpSkin = _getParamSkin(_plugin['MP Skin Settings'] || '{}');
    var _tpSkin = _getParamSkin(_plugin['TP Skin Settings'] || '{}');
    var _stateSkin = _getStateSkin(_plugin['State Skin Settings'] || '{}');

    var _imgTargetCursor = _plugin['Target Cursor Image'] || '';
    var _imgStatusSkin = _plugin['Status Skin Image'] || '';
    var _imgStatusCursor = _plugin['Status Cursor Image'] || '';

    var _adjustPositionX = Number(_plugin['Adjust Position X'] || 0);
    var _adjustPositionY = Number(_plugin['Adjust Position Y'] || 0);

    //=========================================================================
    // 継承元の変更機能
    //=========================================================================

    var _changeSuperClass = function (_class, _super) {
        var oldPrototype = _class.prototype;
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;
        for (var property in oldPrototype) {
            if (oldPrototype.hasOwnProperty(property)) {
                _class.prototype[property] = oldPrototype[property];
            }
        }
    };

    //=========================================================================
    // Scene_Boot
    //=========================================================================

    (function (_class, _super) {

        // ● システム画像のロード
        var _loadSystemImages = _class.loadSystemImages;
        _class.loadSystemImages = function () {
            _loadSystemImages.call(this);
            ImageManager.reserveSystem(_imgTargetCursor);
            ImageManager.reserveSystem(_imgStatusSkin);
            ImageManager.reserveSystem(_imgStatusCursor);
            ImageManager.reserveSystem(_hpSkin.numberImage);
            ImageManager.reserveSystem(_hpSkin.gaugeImage);
            ImageManager.reserveSystem(_mpSkin.numberImage);
            ImageManager.reserveSystem(_mpSkin.gaugeImage);
            ImageManager.reserveSystem(_tpSkin.numberImage);
            ImageManager.reserveSystem(_tpSkin.gaugeImage);
        };

    })(Scene_Boot, Scene_Base);

    //=========================================================================
    // BattleManager
    //=========================================================================

    (function (_class, _super) {

        // ● アクターレイヤーの取得
        _class.actorLayer = function (actor) {
            return this._statusWindow.actorLayer(actor);
        };

        // ● エネミーレイヤーの取得
        _class.enemyLayer = function (enemy) {
            return this._spriteset.enemyLayer(enemy);
        };

        // ● ターゲットレイヤーの取得
        _class.targetLayer = function (target) {
            if (target.isActor()) {
                return this.actorLayer(target);
            } else if (target.isEnemy()) {
                return this.enemyLayer(target);
            }
            return Rectangle.emptyRectangle;
        };

    })(BattleManager, null);

    //=========================================================================
    // Sprite_Battler
    //=========================================================================

    (function (_class, _super) {

        // ● ターゲットの判定
        _class.prototype.isTarget = function (battler) {
            return this._battler === battler;
        };

        // ● 表示領域の取得
        _class.prototype.displayRect = function () {
            var x = this._homeX - this.width * this.anchor.x;
            var y = this._homeY - this.height * this.anchor.y;
            var width = this.width;
            var height = this.height;
            return new Rectangle(x, y, width, height);
        };

    })(Sprite_Battler, Sprite_Base);

    //=========================================================================
    // Sprite_Actor
    //=========================================================================

    (function (_class, _super) {

        // ● 基本位置の設定
        var _setActorHome = _class.prototype.setActorHome;
        _class.prototype.setActorHome = function (index) {
            _setActorHome.call(this, index);
            if (_adjustPositionX !== 0 || _adjustPositionY !== 0) {
                var x = this._homeX + _adjustPositionX;
                var y = this._homeY + _adjustPositionY;
                this.setHome(x, y);
            }
        };

    })(Sprite_Actor, Sprite_Battler);

    //=========================================================================
    // Spriteset_Battle
    //=========================================================================

    (function (_class, _super) {

        // ● エネミーレイヤーの取得
        _class.prototype.enemyLayer = function (enemy) {
            for (var i = 0; i < this._enemySprites.length; i++) {
                var sprite = this._enemySprites[i];
                if (sprite.isTarget(enemy)) {
                    return sprite.displayRect();
                }
            }
            return null;
        };

    })(Spriteset_Battle, Spriteset_Base);

    //=========================================================================
    // Sprite_BattleTarget
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Sprite_BattleTarget() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.Sprite_Drawing;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Sprite_BattleTarget = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function () {
            _super.prototype.initialize.call(this, 0, 0, 32, 32);
            this._battler = null;
            this._selected = false;
            this.createCursorSprite();
        };

        // ● フレーム更新
        _class.prototype.update = function () {
            _super.prototype.update.call(this);
            this.updateCursorState();
        };

        // ● セットアップ
        _class.prototype.setup = function (battler) {
            this._battler = battler;
            this.updateSelectedRect();
            this.updateCursorPosition();
        };

        //---------------------------------------------------------------------
        // 状態処理
        //---------------------------------------------------------------------

        // ● 選択状態にする
        _class.prototype.select = function () {
            this._selected = true;
            this.updateCursorState();
        };

        // ● 非選択状態にする
        _class.prototype.deselect = function () {
            this._selected = false;
            this.updateCursorState();
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● カーソル画像の作成
        _class.prototype.createCursorSprite = function () {
            this._cursorSprite = new $.Sprite_TargetCursor();
            this.addChild(this._cursorSprite);
        };

        // ● カーソル状態の更新
        _class.prototype.updateCursorState = function () {
            if (this._battler && this._battler.isSelected()) {
                this._cursorSprite.visible = true;
            } else {
                this._cursorSprite.visible = false;
            }
            if (this._selected) {
                this._cursorSprite.setCursorIndex(0);
            } else {
                this._cursorSprite.setCursorIndex(1);
            }
        };

        // ● カーソル位置の更新
        _class.prototype.updateCursorPosition = function () {
            this._cursorSprite.x = this.width / 2;
            this._cursorSprite.y = 0;
        };

        // ● 選択範囲の更新
        _class.prototype.updateSelectedRect = function () {
            if (this._battler) {
                var layer = BattleManager.targetLayer(this._battler);
                this.move(layer.x, layer.y);
                this.setFrame(0, 0, layer.width, layer.height);
            }
        };

    })();

    //=========================================================================
    // Sprite_TargetCursor
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Sprite_TargetCursor() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = Sprite;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Sprite_TargetCursor = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function () {
            _super.prototype.initialize.call(this);
            this.initMembers();
            this.loadBitmap();
        };

        // ● 変数の初期化
        _class.prototype.initMembers = function () {
            this._cursorIndex = 0;
            this._animationCount = 0;
            this.anchor.x = 0.5;
            this.anchor.y = 0.5;
            this.visible = false;
        };

        // ● フレーム更新
        _class.prototype.update = function () {
            _super.prototype.update.call(this);
            this._animationCount++;
            this.updateFrame();
        };

        //---------------------------------------------------------------------
        // 状態処理
        //---------------------------------------------------------------------

        // ● カーソル番号の設定
        _class.prototype.setCursorIndex = function (index) {
            this._cursorIndex = index;
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● ビットマップのロード
        _class.prototype.loadBitmap = function () {
            this.bitmap = ImageManager.loadSystem(_imgTargetCursor);
            this.setFrame(0, 0, 0, 0);
        };

        // ● フレームの大きさの取得
        _class.prototype.frameSize = function () {
            return this.bitmap.height / 2;
        };

        // ● アニメーションの間隔
        _class.prototype.animationWait = function () {
            return 4;
        };

        // ● フレームの更新
        _class.prototype.updateFrame = function () {
            var size = this.frameSize();
            var fi = this.animationWait();
            var fs = this.bitmap.width / size;
            var px = size * Math.floor(this._animationCount / fi % fs);
            var py = size * this._cursorIndex;
            this.setFrame(px, py, size, size);
        };

    })();

    //=========================================================================
    // Perch_BattleTarget
    // 　エネミーやアクターを選択するクラスです。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Perch_BattleTarget() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.Perch_Selection;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Perch_BattleTarget = _class;

        // ● 機能拡張
        $.EXPerch_DistanceSelection(_class, _super); // 距離選択
        $.EXPerch_TouchSelection(_class, _super); // タッチ選択
        $.EXPerch_CommandSprites(_class, _super); // コマンド画像

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function () {
            var width = Graphics.boxWidth;
            var height = Graphics.boxHeight;
            _super.prototype.initialize.call(this, 0, 0, width, height);
            this._targets = [];
            this._inputLock = false;
            this._selectDead = false;
            this.createSprites();
            this.hide();
        };

        // ● リフレッシュ
        _class.prototype.refresh = function () {
            this.refreshSprites();
        };

        // ● 位置の更新
        _class.prototype.updatePosition = function () {
            _super.prototype.updatePosition.call(this);
            this.updateSpriteSelection();
        };

        // ● ターゲットの取得
        _class.prototype.target = function () {
            return this._targets[this.index()];
        };

        //---------------------------------------------------------------------
        // 項目処理
        //---------------------------------------------------------------------

        // ● インデックスの取得
        _class.prototype.index = function () {
            return this.selector();
        };

        // ● 要素数の取得
        _class.prototype.maxItems = function () {
            return this._targets.length;
        };

        //---------------------------------------------------------------------
        // カーソル処理
        //---------------------------------------------------------------------

        // ● カーソル移動可能判定
        _class.prototype.isCursorMovable = function () {
            return !this._inputLock && _super.prototype.isCursorMovable.call(this);
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● カーソルビットマップのロード
        _class.prototype.loadCursorBitmap = function () {
            this.cursorBitmap = ImageManager.loadEmptyBitmap();
        };

        // ● コマンド画像の生成
        _class.prototype.generateCommandSprite = function () {
            return new $.Sprite_BattleTarget();
        };

        // ● 画像の作成
        _class.prototype.createSprites = function () {
            this.clearCommandSprites();
        };

        // ● 画像のリフレッシュ
        _class.prototype.refreshSprites = function () {
            this.resizeCommandSprites(this.maxItems());
            for (var i = 0; i < this.maxItems(); i++) {
                this._commandSprites[i].setup(this._targets[i]);
            }
        };

        // ● 画像選択の更新
        _class.prototype.updateSpriteSelection = function () {
            var sprites = this._commandSprites;
            for (var i = 0; i < sprites.length; i++) {
                if (this.index() === i) {
                    sprites[i].select();
                } else {
                    sprites[i].deselect();
                }
            }
        };

    })();

    //=========================================================================
    // Window_BattleActor
    //=========================================================================

    (function (_class, _super) {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● 機能拡張
        _changeSuperClass(_class, _super); // 継承元変更

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ★ 初期化 (再定義)
        _class.prototype.initialize = function () {
            _super.prototype.initialize.call(this);
        };

        // ● リフレッシュ
        _class.prototype.refresh = function () {
            this.makeTargets();
            _super.prototype.refresh.call(this);
        };

        // ● ターゲットの作成
        _class.prototype.makeTargets = function () {
            this._targets = $gameParty.members();
        };

        // ★ アクターの取得 (再定義)
        _class.prototype.actor = function () {
            return this.target();
        };

        //---------------------------------------------------------------------
        // 状態処理
        //---------------------------------------------------------------------

        // ★ 表示 (再定義)
        _class.prototype.show = function () {
            this.select(0);
            _super.prototype.show.call(this);
        };

        // ★ 非表示 (再定義)
        _class.prototype.hide = function () {
            _super.prototype.hide.call(this);
            $gameParty.select(null);
        };

        //---------------------------------------------------------------------
        // 項目処理
        //---------------------------------------------------------------------

        // ★ 項目の選択 (再定義)
        _class.prototype.select = function (index) {
            _super.prototype.select.call(this, index);
            $gameParty.select(this.actor());
        };

        //---------------------------------------------------------------------
        // ハンドラ処理
        //---------------------------------------------------------------------

        // ● 決定可能判定
        _class.prototype.isOkEnabled = function () {
            return _super.prototype.isOkEnabled.call(this);
        };

        // ● 現在の項目の選択可能判定
        _class.prototype.isCurrentItemEnabled = function () {
            if (this._selectDead) {
                return this.actor().isDead();
            }
            return _super.prototype.isCurrentItemEnabled.call(this);
        };

        //---------------------------------------------------------------------
        // タッチ処理
        //---------------------------------------------------------------------

        // ● タッチ処理
        _class.prototype.processTouch = function () {
            _super.prototype.processTouch.call(this);
        };

        //---------------------------------------------------------------------
        // 互換処理
        //---------------------------------------------------------------------

        // ● 自動選択 (YEP_BattleEngineCore)
        var _autoSelect = _class.prototype.autoSelect;
        _class.prototype.autoSelect = function () {
            _autoSelect.call(this);
            var action = BattleManager.inputtingAction();
            if (action && action.isForDeadFriend() && action.isForAll()) {
                this._inputLock = false;
            }
        };

    })(Window_BattleActor, $.Perch_BattleTarget);

    //=========================================================================
    // Window_BattleEnemy
    //=========================================================================

    (function (_class, _super) {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● 機能拡張
        _changeSuperClass(_class, _super); // 継承元変更

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ★ 初期化 (再定義)
        _class.prototype.initialize = function () {
            _super.prototype.initialize.call(this);
            this._enemies = [];
        };

        // ★ リフレッシュ (再定義)
        _class.prototype.refresh = function () {
            this.makeTargets();
            _super.prototype.refresh.call(this);
        };

        // ● 許可されたターゲットの取得
        _class.prototype.allowedTargets = function () {
            return $gameTroop.aliveMembers();
        };

        // ● ターゲットの作成
        _class.prototype.makeTargets = function () {
            this._targets = this.allowedTargets();
            this._enemies = this._targets;
        };

        // ★ エネミーの取得 (再定義)
        _class.prototype.enemy = function () {
            return this.target();
        };

        //---------------------------------------------------------------------
        // 状態処理
        //---------------------------------------------------------------------

        // ★ 表示 (再定義)
        _class.prototype.show = function () {
            this.select(0);
            _super.prototype.show.call(this);
        };

        // ★ 非表示 (再定義)
        _class.prototype.hide = function () {
            _super.prototype.hide.call(this);
            $gameTroop.select(null);
        };

        //---------------------------------------------------------------------
        // 項目処理
        //---------------------------------------------------------------------

        // ★ 項目の選択 (再定義)
        _class.prototype.select = function (index) {
            _super.prototype.select.call(this, index);
            $gameTroop.select(this.enemy());
        };

        // ★ 要素数の取得 (再定義)
        _class.prototype.maxItems = function () {
            return this._targets.length;
        };

        //---------------------------------------------------------------------
        // タッチ処理
        //---------------------------------------------------------------------

        // ● タッチ処理
        _class.prototype.processTouch = function () {
            _super.prototype.processTouch.call(this);
        };

    })(Window_BattleEnemy, $.Perch_BattleTarget);

    //=========================================================================
    // Layer_OneBattlerStatus
    // 　バトルステータス用の1キャラ分のレイヤーです。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Layer_OneBattlerStatus() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.LayerContainer;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Layer_OneBattlerStatus = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function () {
            _super.prototype.initialize.call(this);
            this.move(0, 0, this.layerWidth(), this.layerHeight());
            this._actor = null;
            this.createSprites();
        };

        // ● アクターの設定
        _class.prototype.setActor = function (actor) {
            if (this._actor !== actor) {
                this._actor = actor;
                this.refresh();
            }
        };

        // ● リフレッシュ
        _class.prototype.refresh = function () {
            this.refreshSprites();
            this.refreshVisibility();
        };

        // ● フレーム更新
        _class.prototype.update = function () {
            _super.prototype.update.call(this);
            this.updatePlacement();
        };

        // ● 表示領域の取得
        _class.prototype.displayRect = function () {
            var x = this.absoluteX();
            var y = this.absoluteY();
            var width = this.width;
            var height = this.height;
            return new Rectangle(x, y, width, height);
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● 内容の作成
        _class.prototype.createSprites = function () {
            var w = this.width;
            var h = this.height;
            this._graphicSprite = new $.Sprite_Drawing(0, 0, w, h);
            this._skinSprite = new $.Sprite_Drawing(0, 0, w, h);
            this._stateSprite = new Sprite_StateIcon();
            this.addChild(this._graphicSprite);
            this.addChild(this._skinSprite);
            this.addChild(this._stateSprite);
        };

        // ● 画像のリフレッシュ
        _class.prototype.refreshSprites = function () {
            this.refreshGraphic();
            this.refreshSkin();
            this.refreshStates();
        };

        // ● 表示状態のリフレッシュ
        _class.prototype.refreshVisibility = function () {
            if (this._actor) {
                this.visible = true;
            } else {
                this.visible = false;
            }
        };

        //---------------------------------------------------------------------
        // 配置処理
        //---------------------------------------------------------------------

        // ● レイヤー幅の取得
        _class.prototype.layerWidth = function () {
            return ImageManager.loadSystem(_imgStatusSkin).width;
        };

        // ● レイヤー高さの取得
        _class.prototype.layerHeight = function () {
            return ImageManager.loadSystem(_imgStatusSkin).height;
        };

        // ● 配置の更新
        _class.prototype.updatePlacement = function () {
            if (this._actor) {
                var perch = this.parent;
                var index = this._actor.index();
                var max = $gameParty.battleMembers().length;
                this.x = perch.width / 2 + this.width * (index - max / 2);
                this.y = perch.height - this.height * perch.openness / 255;
            }
        };

        //---------------------------------------------------------------------
        // グラフィック処理
        //---------------------------------------------------------------------

        // ● 戦闘不能時のトーンの取得
        _class.prototype.deadGraphicTone = function () {
            return [0, 0, 0, 255];
        };

        // ● ピンチ時のトーンの取得
        _class.prototype.dyingGraphicTone = function () {
            return [64, -32, -32, 0];
        };

        // ● グラフィックのリフレッシュ
        _class.prototype.refreshGraphic = function () {
            this._graphicSprite.contents.clear();
            if (this._actor) {
                this.drawBattlerGraphic();
                this.setBattlerGraphicTone();
            }
        };

        // ● グラフィックの描画
        _class.prototype.drawBattlerGraphic = function () {
            if (_actorSkin.graphicType === 'face') {
                var faceName = this._actor.faceName();
                var faceIndex = this._actor.faceIndex();
                var x = _actorSkin.graphicX;
                var y = _actorSkin.graphicY;
                this._graphicSprite.drawFace(faceName, faceIndex, x, y);
            }
        };

        // ● グラフィックのトーン設定
        _class.prototype.setBattlerGraphicTone = function () {
            if (this._actor.isDead()) {
                return this._graphicSprite.setColorTone(this.deadGraphicTone());
            } else if (this._actor.isDying()) {
                return this._graphicSprite.setColorTone(this.dyingGraphicTone());
            } else {
                return this._graphicSprite.setColorTone([0, 0, 0, 0]);
            }
        };

        //---------------------------------------------------------------------
        // スキン処理
        //---------------------------------------------------------------------

        // ● スキンのリフレッシュ
        _class.prototype.refreshSkin = function () {
            this._skinSprite.contents.clear();
            if (this._actor) {
                this.drawBattlerSkin();
                this.drawBattlerName();
                this.drawBattlerParam(_hpSkin, this._actor.hp, this._actor.mhp);
                this.drawBattlerParam(_mpSkin, this._actor.mp, this._actor.mmp);
                this.drawBattlerParam(_tpSkin, this._actor.tp, this._actor.maxTp());
            }
        };

        // ● スキンの描画
        _class.prototype.drawBattlerSkin = function () {
            var bitmap = ImageManager.loadSystem(_imgStatusSkin);
            var w = bitmap.width;
            var h = bitmap.height;
            this._skinSprite.blt(bitmap, 0, 0, w, h, 0, 0, w, h);
        };

        // ● 名前の描画
        _class.prototype.drawBattlerName = function () {
            var size = _actorSkin.nameSize;
            if (size > 0) {
                var x = _actorSkin.nameX;
                var y = _actorSkin.nameY;
                var w = _actorSkin.nameWidth;
                var h = size + 8;
                var a = _actorSkin.nameAlign;
                this._skinSprite.resetTextColor();
                this._skinSprite.contents.fontSize = size;
                this._skinSprite.contents.drawText(this._actor.name(), x, y, w, h, a);
            }
        };

        // ● パラメータの描画
        _class.prototype.drawBattlerParam = function (skin, current, max) {
            var rate = current / Math.max(max, 1);
            if (skin.gaugeImage) {
                this.drawBattlerGauge(skin.gaugeImage, skin.gaugeX, skin.gaugeY, rate);
            }
            if (skin.numberImage) {
                this.drawBattlerNumber(skin.numberImage, skin.numberX, skin.numberY, current, rate);
            }
        };

        // ● ゲージの描画
        _class.prototype.drawBattlerGauge = function (fileName, x, y, rate) {
            var bitmap = ImageManager.loadSystem(fileName);
            var w = Math.floor(bitmap.width * rate);
            var h = bitmap.height;
            this._skinSprite.blt(bitmap, 0, 0, w, h, x, y, w, h);
        };

        // ● 数値の描画
        _class.prototype.drawBattlerNumber = function (fileName, x, y, value, rate) {
            var bitmap = ImageManager.loadSystem(fileName);
            var string = Math.abs(value).toString();
            var s = string.length;
            var w = bitmap.width / 10;
            var h = bitmap.height / 4;
            var c = 0;
            if (rate > 0.5) {
                c = 0;
            } else if (rate > 0.25) {
                c = h;
            } else if (rate > 0) {
                c = h * 2;
            } else {
                c = h * 3;
            }
            for (var i = 0; i < s; i++) {
                var n = Number(string[i]);
                this._skinSprite.blt(bitmap, n * w, c, w, h, x + (i - s) * w, y, w, h);
            }
        };

        //---------------------------------------------------------------------
        // ステート処理
        //---------------------------------------------------------------------

        // ● ステートのリフレッシュ
        _class.prototype.refreshStates = function () {
            this._stateSprite.setup(this._actor);
            this._stateSprite.move(_stateSkin.iconX, _stateSkin.iconY);
        };

    })();

    //=========================================================================
    // Window_BattleStatus
    //=========================================================================

    (function (_class, _super) {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● 機能拡張
        _changeSuperClass(_class, _super); // 継承元変更
        $.EXPerch_Compatible(_class, _super); // 互換拡張

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ★ 初期化 (再定義)
        _class.prototype.initialize = function () {
            var x = this.perchX();
            var y = this.perchY();
            var width = this.perchWidth();
            var height = this.perchHeight();
            _super.prototype.initialize.call(this, x, y, width, height);
            this.createSprites();
            this.refresh();
            this.openness = 0;
        };

        // ★ リフレッシュ (再定義)
        _class.prototype.refresh = function () {
            this.refreshSprites();
        };

        // ● フレームの更新
        _class.prototype.update = function () {
            _super.prototype.update.call(this);
            this.updatePosition();
        };

        // ● アクターレイヤーの取得
        _class.prototype.actorLayer = function (actor) {
            var layer = this._statusLayers[actor.index()];
            if (layer) {
                return layer.displayRect();
            }
            return null;
        };

        //---------------------------------------------------------------------
        // 項目処理
        //---------------------------------------------------------------------

        // ★ 要素数の取得 (再定義)
        _class.prototype.maxItems = function () {
            return $gameParty.maxBattleMembers();
        };

        // ● 選択可能なオブジェクト配列を取得
        _class.prototype.selectableObjects = function () {
            return this._statusLayers;
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● カーソルビットマップのロード
        _class.prototype.loadCursorBitmap = function () {
            this._cursorPosition = 'center';
            this._cursorOffsetX = 0;
            this._cursorOffsetY = 0;
            this._cursorDisplay = 'behind';
            this.cursorBitmap = ImageManager.loadSystem(_imgStatusCursor);
        };

        // ● 画像の作成
        _class.prototype.createSprites = function () {
            this._statusLayers = [];
            for (var i = 0; i < this.maxItems(); i++) {
                var layer = new $.Layer_OneBattlerStatus();
                this._statusLayers.push(layer);
                this.addChild(layer);
            }
            this.createDummySprite();
        };

        // ● 画像のリフレッシュ
        _class.prototype.refreshSprites = function () {
            var actors = $gameParty.battleMembers();
            for (var i = 0; i < this._statusLayers.length; i++) {
                var layer = this._statusLayers[i];
                layer.setActor(actors[i]);
                layer.refresh();
            }
        };

        // ● 項目の描画
        _class.prototype.drawItem = function (index) {
            var layer = this._statusLayers[index];
            if (layer) {
                layer.refresh();
            }
        };

        //---------------------------------------------------------------------
        // 配置処理
        //---------------------------------------------------------------------

        // ● パーチのX座標の取得
        _class.prototype.perchX = function () {
            return 0;
        };

        // ● パーチのY座標の取得
        _class.prototype.perchY = function () {
            return Graphics.boxHeight - this.perchHeight();
        };

        // ● パーチの幅の取得
        _class.prototype.perchWidth = function () {
            return Graphics.boxWidth;
        };

        // ● パーチの高さの取得
        _class.prototype.perchHeight = function () {
            return ImageManager.loadSystem(_imgStatusSkin).height;
        };

    })(Window_BattleStatus, $.Perch_Selection);

    //=========================================================================
    // Window_PartyCommand
    //=========================================================================

    (function (_class, _super) {

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● セットアップ
        var _setup = _class.prototype.setup;
        _class.prototype.setup = function () {
            _setup.call(this);
            this.updatePlacement();
        };

        //---------------------------------------------------------------------
        // 配置処理
        //---------------------------------------------------------------------

        // ● 位置の更新
        _class.prototype.updatePlacement = function () {
            this.x = 0;
            this.y = 0;
        };

        // ★ ウィンドウ幅の取得 (再定義)
        _class.prototype.windowWidth = function () {
            return Graphics.boxWidth;
        };

        // ★ 表示する行数の取得 (再定義)
        _class.prototype.numVisibleRows = function () {
            return 1;
        };

        // ★ 列数の取得 (再定義)
        _class.prototype.maxCols = function () {
            return this.maxItems();
        };

        // ★ 文字位置の取得 (再定義)
        _class.prototype.itemTextAlign = function () {
            return 'center';
        };

    })(Window_PartyCommand, Window_Command);

    //=========================================================================
    // Window_ActorCommand
    //=========================================================================

    (function (_class, _super) {

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● セットアップ
        var _setup = _class.prototype.setup;
        _class.prototype.setup = function (actor) {
            _setup.call(this, actor);
            this.updatePlacement();
        };

        //---------------------------------------------------------------------
        // 状態処理
        //---------------------------------------------------------------------

        // ★ アクティブ化 (再定義)
        _class.prototype.activate = function () {
            _super.prototype.activate.call(this);
            this.open();
        };

        // ★ 非アクティブ化 (再定義)
        _class.prototype.deactivate = function () {
            _super.prototype.deactivate.call(this);
            this.close();
        };

        //---------------------------------------------------------------------
        // 配置処理
        //---------------------------------------------------------------------

        // ● 位置の更新
        _class.prototype.updatePlacement = function () {
            if (this._actor) {
                var layer = BattleManager.actorLayer(this._actor);
                var x = layer.x + layer.width / 2 - this.width / 2;
                var y = layer.y - this.height;
                this.x = x.clamp(0, Graphics.boxWidth - this.width);
                this.y = y.clamp(0, Graphics.boxHeight - this.height);
            }
        };

    })(Window_ActorCommand, Window_Command);

    //=========================================================================
    // Scene_Battle
    //=========================================================================

    (function (_class, _super) {

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● シーンの開始
        var _start = _class.prototype.start;
        _class.prototype.start = function () {
            _start.call(this);
            this.refreshStatus();
        };

        // ★ ウィンドウ位置を更新 (再定義)
        _class.prototype.updateWindowPositions = function () {
            // メモ : 位置を更新しない！
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● ステータスウィンドウの作成
        var _createStatusWindow = _class.prototype.createStatusWindow;
        _class.prototype.createStatusWindow = function () {
            _createStatusWindow.call(this);
            this.addPerch(this._statusWindow);
        };

    })(Scene_Battle, Scene_Base);

})(Vitsuno);
