//=============================================================================
// Vitsuno_FrontBattler.js
//=============================================================================
// [更新情報]
// Ver 1.02 : デフォルト値の変更。
// Ver 1.01 : 「表示の更新」の定義方法を変更。
// Ver 1.00 : 基本版完成！
//-----------------------------------------------------------------------------
// Copyright (c) 2018 Tsuno Ozumi
// Released under the MIT license
// https://opensource.org/licenses/mit-license.php
//-----------------------------------------------------------------------------
// Site : https://vitsuno.net/
//=============================================================================

/*:
 * @plugindesc ＜バトル＞フロントビューでアクターグラフィックを表示します。
 * @author 尾角 つの (Tsuno Ozumi)
 *
 * @param (Basic Settings)
 * @type note
 * @default ""
 *
 * @param Graphic Type
 * @desc グラフィックの種類
 * @type select
 * @option 顔グラフィック
 * @value face
 * @option サイドビュー
 * @value sideview
 * @option なし
 * @value none
 * @default face
 *
 * @param Hide With Status
 * @desc ステータスウィンドウに合わせて隠すか？
 * @type boolean
 * @default true
 *
 * @param Hide On Death
 * @desc 先頭不能時に隠すか？
 * @type boolean
 * @default true
 *
 * @param Actor Interval
 * @desc アクター間の幅
 * @type number
 * @default 200
 *
 * @param
 *
 * @param (Advanced Settings)
 * @type note
 * @default ""
 *
 * @param Arrangement
 * @desc グラフィックの配置
 * @type struct<arrangement>
 * @default {"Actor X":"0","Actor Y":"-58","Damage X":"0","Damage Y":"-96","Start Position":"{\"X\":\"0\",\"Y\":\"0\",\"Duration\":\"0\"}","Entry Position":"{\"X\":\"0\",\"Y\":\"0\",\"Duration\":\"30\"}","Forward Position":"{\"X\":\"0\",\"Y\":\"-18\",\"Duration\":\"9\"}","Back Position":"{\"X\":\"0\",\"Y\":\"0\",\"Duration\":\"9\"}","Retreat Position":"{\"X\":\"0\",\"Y\":\"0\",\"Duration\":\"30\"}","Refresh Position":"{\"X\":\"0\",\"Y\":\"0\",\"Duration\":\"0\"}"}
 *
 * @help
 * 
 * フロントビューでアクターのグラフィックを表示し、
 * アニメーションとダメージポップアップの演出を行います。
 *
 * ＜詳細と注意点＞
 * ・ステータス変化系のプラグインと組み合わせて使用してください。
 */

/*~struct~arrangement:
 *
 * @param Actor X
 * @desc アクター位置のX座標の調整
 * @type number
 * @max 9999
 * @min -9999
 *
 * @param Actor Y
 * @desc アクター位置のY座標の調整
 * @type number
 * @max 9999
 * @min -9999
 *
 * @param Damage X
 * @desc ダメージ位置のX座標の調整
 * @type number
 * @max 9999
 * @min -9999
 *
 * @param Damage Y
 * @desc ダメージ位置のY座標の調整
 * @type number
 * @max 9999
 * @min -9999
 *
 * @param Start Position
 * @desc バトル開始時の位置
 * @type struct<stepMove>
 *
 * @param Entry Position
 * @desc 参入時の位置
 * @type struct<stepMove>
 *
 * @param Forward Position
 * @desc 前進時の位置
 * @type struct<stepMove>
 *
 * @param Back Position
 * @desc 後退時の位置
 * @type struct<stepMove>
 *
 * @param Retreat Position
 * @desc 退却時の位置
 * @type struct<stepMove>
 *
 * @param Refresh Position
 * @desc リフレッシュ時の位置
 * @type struct<stepMove>
 *
 */

/*~struct~stepMove:
 *
 * @param X
 * @desc X座標
 * @type number
 * @max 9999
 * @min -9999
 *
 * @param Y
 * @desc Y座標
 * @type number
 * @max 9999
 * @min -9999
 *
 * @param Duration
 * @desc 動作フレーム数
 * @type number
 *
 */

var Vitsuno = Vitsuno || {};

(function ($) {
    'use strict';

    // バージョン情報
    $.FRONT_BATTLER_VERSION = 1.02;

    // プラグインパラメータの処理
    var _getStepMove = function (json) {
        var data = JSON.parse(json);
        var step = {};
        step.x = Number(data['X'] || 0);
        step.y = Number(data['Y'] || 0);
        step.duration = Number(data['Duration'] || 0);
        return step;
    };

    var _getArrangement = function (json) {
        var data = JSON.parse(json);
        var arr = {};

        arr.actorX = Number(data['Actor X'] || 0);
        arr.actorY = Number(data['Actor Y'] || 0);
        arr.damageX = Number(data['Damage X'] || 0);
        arr.damageY = Number(data['Damage Y'] || 0);

        arr.startStep = _getStepMove(data['Start Position'] || '{}');
        arr.entryStep = _getStepMove(data['Entry Position'] || '{}');
        arr.forwardStep = _getStepMove(data['Forward Position'] || '{}');
        arr.backStep = _getStepMove(data['Back Position'] || '{}');
        arr.retreatStep = _getStepMove(data['Retreat Position'] || '{}');
        arr.refreshStep = _getStepMove(data['Refresh Position'] || '{}');

        return arr;
    };

    // プラグインの設定値を取得
    var _plugin = PluginManager.parameters('Vitsuno_FrontBattler');

    var _graphicType = _plugin['Graphic Type'] || 'none';
    var _hideWidhStatus = _plugin['Hide With Status'] === 'true';
    var _hideOnDeath = _plugin['Hide On Death'] === 'true';
    var _actorInterval = Number(_plugin['Actor Interval'] || 0);

    var _arrangement = _getArrangement(_plugin['Arrangement'] || '{}');

    //=========================================================================
    // BattleManager
    //=========================================================================

    (function (_class, _super) {

        // ● ステータスのリフレッシュ
        var _refreshStatus = _class.refreshStatus;
        _class.refreshStatus = function () {
            _refreshStatus.call(this);
            this._spriteset.refreshStatus();
        };

        // ● ステータスウィンドウの表示状態の取得
        _class.isStatusWindowVisible = function () {
            return this._statusWindow && this._statusWindow.isOpen();
        };

    })(BattleManager, null);

    //=========================================================================
    // Game_Actor
    //=========================================================================

    (function (_class, _super) {

        // ● バトラータイプの取得
        _class.prototype.battlerType = function () {
            return _graphicType;
        };

        // ★ スプライトの表示判定 (再定義)
        _class.prototype.isSpriteVisible = function () {
            return true;
        };

    })(Game_Actor, Game_Battler);

    //=========================================================================
    // Sprite_Actor
    //=========================================================================

    (function (_class, _super) {

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 変数の初期化
        var _initMembers = _class.prototype.initMembers;
        _class.prototype.initMembers = function () {
            _initMembers.call(this);
            this._battlerType = 'none';
        };

        // ● フレーム更新
        var _update = _class.prototype.update;
        _class.prototype.update = function () {
            _update.call(this);
            this.updateWeaponVisibility();
            this.updateStateVisibility();
        };

        // ★ 表示の更新 (再定義)
        _class.prototype.updateVisibility = function () {
            _super.prototype.updateVisibility.call(this);
            if (_hideWidhStatus && !BattleManager.isStatusWindowVisible()) {
                this.visible = false;
            }
            if (_hideOnDeath && this._actor && this._actor.isDead()) {
                this.visible = false;
            }
        };

        // ● ステータスのリフレッシュ
        _class.prototype.refreshStatus = function () {
            if (this._actor) {
                this.setActorHome(this._actor.index());
            }
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● 顔グラフィックの判定
        _class.prototype.isFaceBattler = function () {
            return this._battlerType === 'face';
        };

        // ● サイドビューグラフィックの判定
        _class.prototype.isSideViewBattler = function () {
            return this._battlerType === 'sideview';
        };

        // ● タイプ名からバトラー名の取得
        _class.prototype.battlerNameByType = function (type) {
            if (type === 'face') {
                return this._actor.faceName();
            } else if (type === 'sideview') {
                return this._actor.battlerName();
            }
            return '';
        };

        // ★ ビットマップの更新 (再定義)
        _class.prototype.updateBitmap = function () {
            _super.prototype.updateBitmap.call(this);
            var type = this._actor.battlerType();
            var name = this.battlerNameByType(type);
            if (this._battlerType !== type || this._battlerName !== name) {
                this._battlerType = type;
                this._battlerName = name;
                this.updateMainBitmap();
            }
        };

        // ● メインビットマップの更新
        _class.prototype.updateMainBitmap = function () {
            var name = this._battlerName;
            if (this.isFaceBattler()) {
                this._mainSprite.bitmap = ImageManager.loadFace(name);
            } else if (this.isSideViewBattler()) {
                this._mainSprite.bitmap = ImageManager.loadSvActor(name);
            } else {
                this._mainSprite.bitmap = null;
            }
        };

        // ★ フレームの更新 (再定義)
        _class.prototype.updateFrame = function () {
            _super.prototype.updateFrame.call(this);
            if (this._mainSprite.bitmap) {
                this.updateMainFrame();
            }
        };

        // ● メインフレームの更新
        _class.prototype.updateMainFrame = function () {
            if (this.isFaceBattler()) {
                this.updateFaceFrame();
            } else if (this.isSideViewBattler()) {
                this.updateSideViewFrame();
            }
        };

        // ● 顔グラフィックフレームの更新
        _class.prototype.updateFaceFrame = function () {
            var bitmap = this._mainSprite.bitmap;
            var index = this._actor.faceIndex();
            var sw = Window_Base._faceWidth;
            var sh = Window_Base._faceHeight;
            var sx = index % 4 * sw;
            var sy = Math.floor(index / 4) * sh;
            this._mainSprite.setFrame(sx, sy, sw, sh);
        };

        // ● サイドビューフレームの更新
        _class.prototype.updateSideViewFrame = function () {
            var bitmap = this._mainSprite.bitmap;
            var motionIndex = this._motion ? this._motion.index : 0;
            var pattern = this._pattern < 3 ? this._pattern : 1;
            var cw = bitmap.width / 9;
            var ch = bitmap.height / 6;
            var cx = Math.floor(motionIndex / 6) * 3 + pattern;
            var cy = motionIndex % 6;
            this._mainSprite.setFrame(cx * cw, cy * ch, cw, ch);
        };

        // ● 影の更新
        var _updateShadow = _class.prototype.updateShadow;
        _class.prototype.updateShadow = function () {
            _updateShadow.call(this);
            if (!this.isSideViewBattler()) {
                this._shadowSprite.visible = false;
            }
        };

        // ● 武器表示の更新
        _class.prototype.updateWeaponVisibility = function () {
            if (this.isSideViewBattler()) {
                this._weaponSprite.show();
            } else {
                this._weaponSprite.hide();
            }
        };

        // ● ステート表示の更新
        _class.prototype.updateStateVisibility = function () {
            if (this.isSideViewBattler()) {
                this._stateSprite.show();
            } else {
                this._stateSprite.hide();
            }
        };

        //---------------------------------------------------------------------
        // 移動処理
        //---------------------------------------------------------------------

        // ● 配置の取得
        _class.prototype.battlerArrangement = function () {
            return _arrangement;
        };

        // ★ 基本位置の設定 (再定義)
        var _setActorHome = _class.prototype.setActorHome;
        _class.prototype.setActorHome = function (index) {
            var arr = this.battlerArrangement();
            var max = $gameParty.battleMembers().length;
            var x = Graphics.width / 2 + _actorInterval * (index - (max - 1) / 2);
            var y = Graphics.height;
            this.setHome(x + arr.actorX, y + arr.actorY);
        };

        // ★ ダメージのX座標 (再定義)
        _class.prototype.damageOffsetX = function () {
            return this.battlerArrangement().damageX;
        };

        // ★ ダメージのY座標 (再定義)
        _class.prototype.damageOffsetY = function () {
            return this.battlerArrangement().damageY;
        };

        // ★ 初期位置へ移動 (再定義)
        _class.prototype.moveToStartPosition = function () {
            var step = this.battlerArrangement().startStep;
            this.startMove(step.x, step.y, step.duration);
        };

        // ★ 参入時の操作を開始 (再定義)
        _class.prototype.startEntryMotion = function () {
            if (this._actor && this._actor.canMove()) {
                var step = this.battlerArrangement().entryStep;
                this.startMotion('walk');
                this.startMove(step.x, step.y, step.duration);
            } else if (!this.isMoving()) {
                var step = this.battlerArrangement().refreshStep;
                this.refreshMotion();
                this.startMove(step.x, step.y, step.duration);
            }
        };

        // ★ 前進する (再定義)
        _class.prototype.stepForward = function () {
            var step = this.battlerArrangement().forwardStep;
            this.startMove(step.x, step.y, step.duration);
        };

        // ★ 後退する (再定義)
        _class.prototype.stepBack = function () {
            var step = this.battlerArrangement().backStep;
            this.startMove(step.x, step.y, step.duration);
        };

        // ★ 退却する (再定義)
        _class.prototype.retreat = function () {
            var step = this.battlerArrangement().retreatStep;
            this.startMove(step.x, step.y, step.duration);
        };

    })(Sprite_Actor, Sprite_Battler);

    //=========================================================================
    // Spriteset_Battle
    //=========================================================================

    (function (_class, _super) {

        // ● ステータスのリフレッシュ
        _class.prototype.refreshStatus = function () {
            for (var i = 0; i < this._actorSprites.length; i++) {
                this._actorSprites[i].refreshStatus();
            }
        };

    })(Spriteset_Battle, Spriteset_Base);

    //=========================================================================
    // Scene_Battle
    //=========================================================================

    (function (_class, _super) {

        // ● ステータスのリフレッシュ
        var _refreshStatus = _class.prototype.refreshStatus;
        _class.prototype.refreshStatus = function () {
            _refreshStatus.call(this);
            this._spriteset.refreshStatus();
        };

    })(Scene_Battle, Scene_Base);

})(Vitsuno);
