//=============================================================================
// Vitsuno_SpritePerch.js
//=============================================================================
// [更新情報]
// Ver 4.21 : 画像を使用しないようにデフォルト値を変更。
// Ver 4.20 : 一部ウィンドウ互換拡張機能を追加。
// Ver 4.10 : アイコンゲージ機能を削除、カーソル初期値を固定値化。
// Ver 4.00 : より実用的な仕様に大幅変更。
//          : 細かいパーツとなるクラスを多数追加。
// Ver 3.00 : 使いやすさ向上のため、クラス構成を大幅に刷新！
// Ver 2.01 : カーソル画像のリザーブ機能を追加。
// Ver 2.00 : レイヤー方式に改訂！
//          : pixi.js v4.4.1 (RPGツクールMV Ver 1.50) に対応。
//          : コア機能（クラスの継承、線の描画）を削除。
// Ver 1.01 : RPGツクールMV Ver 1.10 に対応。
// Ver 1.00 : プラグイン完成！
//-----------------------------------------------------------------------------
// Copyright (c) 2015-2018 Tsuno Ozumi
// Released under the MIT license
// https://opensource.org/licenses/mit-license.php
//-----------------------------------------------------------------------------
// Site : https://vitsuno.net/
//=============================================================================

/* <mymemo>
 * 独自クラスは基本 Vitsuno (以下"$") オブジェクトで囲い込んでいます。
 *
 * ＜画像コマンドクラスの作成＞
 * １、$.Perch_Selection を継承します。
 * ２、拡張定義 $.EXPerch_XXX から必要なものを適用します。
 * ３、以下の項目が重要なため再定義を行います。
 * 　initialize : クラス作成時に行う基本処理。主に画像を作成します。（親クラスを継承）
 * 　refresh : シーンからリフレッシュするための基本処理。主に画像を描画します。
 * 　update : 毎フレーム行う基本処理。主に画像を移動します。（親クラスを継承）
 * 　maxItems : 選択可能なコマンドの数を設定。
 * 　selectableObjects : 選択可能なコマンド画像の配列を設定。
 * ４、その他の処理を加えます。
 *
 * ＜描画スプライトについて＞
 * $.Sprite_Drawing は従来の Window_Base の描画処理を元に、
 * 画像側に描画の処理を行わせるために作成したクラスです。例えば、
 * sprite.drawText(text, x, y, width, align);
 * のように、直接スプライトに処理を送って描画します。
 * なお、ビットマップは自動で生成されるので付け直さないでください。
 */

/*:
 * @plugindesc ＜マスター＞画像コマンドを作成するマスタープラグインです。
 * @author 尾角 つの (Tsuno Ozumi)
 *
 * @param (Image Settings)
 * @desc このメモは、ご自由にお使いください。
 * @type note
 * @default ""
 *
 * @param Cursor Image
 * @desc コマンドカーソル画像を指定します。
 * @require 1
 * @dir img/system/
 * @type file
 *
 * @param Command Skin Image
 * @desc コマンドスキン画像を指定します。
 * @require 1
 * @dir img/system/
 * @type file
 *
 * @param Icon Ring Image
 * @desc アイコンリング画像を指定します。
 * @require 1
 * @dir img/system/
 * @type file
 *
 * @param
 *
 * @param (Color Settings)
 * @desc このメモは、ご自由にお使いください。
 * @type note
 * @default ""
 *
 * @param Normal Color
 * @desc 通常の文字色を設定します。
 * @default #ffffff
 *
 * @param System Color
 * @desc システムの文字色を設定します。
 * @default #84aaff
 *
 * @param Crisis Color
 * @desc ピンチ文字色を設定します。
 * @default #ffff40
 *
 * @param Death Color
 * @desc 戦闘不能文字色を設定します。
 * @default #ff2020
 *
 * @param Power Up Color
 * @desc 能力上昇文字色を設定します。
 * @default #80ff80
 *
 * @param Power Down Color
 * @desc 能力減少文字色を設定します。
 * @default #c08080
 *
 * @param MP Cost Color
 * @desc MP消費量の文字色を設定します。
 * @default #40c0f0
 *
 * @param TP Cost Color
 * @desc TP消費量の文字色を設定します。
 * @default #00e060
 *
 * @param Gauge Back Color
 * @desc ゲージ背景色を設定します。
 * @default #202040
 *
 * @param HP Gauge Color 1
 * @desc HPゲージ色１を設定します。
 * @default #e08040
 *
 * @param HP Gauge Color 2
 * @desc HPゲージ色２を設定します。
 * @default #f0c040
 *
 * @param MP Gauge Color 1
 * @desc MPゲージ色１を設定します。
 * @default #4080c0
 *
 * @param MP Gauge Color 2
 * @desc MPゲージ色２を設定します。
 * @default #40c0f0
 *
 * @param TP Gauge Color 1
 * @desc TPゲージ色１を設定します。
 * @default #00a040
 *
 * @param TP Gauge Color 2
 * @desc TPゲージ色２を設定します。
 * @default #00e060
 *
 * @help
 * 
 * 画像コマンドなどの機能を拡張するベースプラグインです。
 * 
 * ＜必要画像＞
 * 　以下のシステム画像を img/system フォルダに入れて下さい。
 *
 * 　PerchCursor.png       // 画像コマンド用カーソル画像
 * 　PerchCommandSkin.png  // コマンドスキン画像
 * 　PerchIconRing.png     // アイコンリング画像
 *
 * ＜詳細と注意点＞
 * ・このプラグイン単体では、特にゲーム上の変化はありません。
 * ・他のプラグインから、このプラグインの機能を呼び出してください。
 * ・システム画像を使用しない場合は、パラメータを（なし）に設定してください。
 */

var Vitsuno = Vitsuno || {};

(function ($) {
    'use strict';

    // バージョン情報
    $.PERCH_VERSION = 4.20;

    // プラグインの設定値を取得
    var _plugin = PluginManager.parameters('Vitsuno_SpritePerch');

    var _imgPerchCursor = _plugin['Cursor Image'] || '';
    var _imgCommandSkin = _plugin['Command Skin Image'] || '';
    var _imgIconRing = _plugin['Icon Ring Image'] || '';

    var _normalColor = _plugin['Normal Color'] || '#ffffff';
    var _systemColor = _plugin['System Color'] || '#84aaff';
    var _crisisColor = _plugin['Crisis Color'] || '#ffff40';
    var _deathColor = _plugin['Death Color'] || '#ff2020';
    var _powerUpColor = _plugin['Power Up Color'] || '#80ff80';
    var _powerDownColor = _plugin['Power Down Color'] || '#c08080';
    var _mpCostColor = _plugin['MP Cost Color'] || '#40c0f0';
    var _tpCostColor = _plugin['TP Cost Color'] || '#00e060';
    var _gaugeBackColor = _plugin['Gauge Back Color'] || '#202040';
    var _hpGaugeColor1 = _plugin['HP Gauge Color 1'] || '#e08040';
    var _hpGaugeColor2 = _plugin['HP Gauge Color 2'] || '#f0c040';
    var _mpGaugeColor1 = _plugin['MP Gauge Color 1'] || '#4080c0';
    var _mpGaugeColor2 = _plugin['MP Gauge Color 2'] || '#40c0f0';
    var _tpGaugeColor1 = _plugin['TP Gauge Color 1'] || '#00a040';
    var _tpGaugeColor2 = _plugin['TP Gauge Color 2'] || '#00e060';

    //=========================================================================
    // DataManager
    //=========================================================================

    (function (_class, _super) {

        // ● データ読み込み時の処理
        var _onLoad = _class.onLoad;
        _class.onLoad = function (object) {
            _onLoad.call(this, object);

            // 対象データの取得
            var items = [$dataSkills, $dataItems, $dataWeapons, $dataArmors];

            // アイコン付きデータの設定
            if (items.contains(object)) {
                object.forEach(function (item) {
                    if (item) {
                        item.meta.exRingIndex = 1;
                        if (item.meta.ringIndex) {
                            item.meta.exRingIndex = Number(item.meta.ringIndex);
                        }
                    }
                });
            }

        };

    })(DataManager, null);

    //=========================================================================
    // Sprite_Drawing
    // 　描画処理を追加したスプライトのクラスです。一部 Window_Base の機能を使用します。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Sprite_Drawing() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = Sprite_Base;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Sprite_Drawing = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● クラス変数
        _class._iconRingWidth = 64;
        _class._iconRingHeight = 64;
        _class._iconGaugeWidth = 32;
        _class._iconGaugeHeight = 32;

        // ● 初期化
        _class.prototype.initialize = function (x, y, width, height) {
            _super.prototype.initialize.call(this);
            this.move(x, y);
            this.createContents(width, height);
            this._touching = false;
            this._clickHandler = null;
        };

        // ● フレーム更新
        _class.prototype.update = function () {
            _super.prototype.update.call(this);
            this.updateProcess();
        };

        // ● 操作処理の更新
        _class.prototype.updateProcess = function () {
            this.processTouch();
        };

        //---------------------------------------------------------------------
        // コンテンツ処理
        //---------------------------------------------------------------------

        // ● コンテンツ
        Object.defineProperty(_class.prototype, 'contents', {
            get: function () {
                return this.bitmap;
            },
            set: function (value) {
                this.bitmap = value;
            },
            configurable: true
        });

        // ● コンテンツの作成
        _class.prototype.createContents = function (width, height) {
            this.contents = new Bitmap(width, height);
            this.resetFontSettings();
        };

        //---------------------------------------------------------------------
        // ハンドラ処理
        //---------------------------------------------------------------------

        // ● ハンドラの設定
        _class.prototype.setClickHandler = function (method) {
            this._clickHandler = method;
        };

        // ● ハンドラの存在判定
        _class.prototype.isClickHandled = function () {
            return !!this._clickHandler;
        };

        // ● ハンドラの呼び出し
        _class.prototype.callClickHandler = function () {
            if (this.isClickHandled()) {
                this._clickHandler();
            }
        };

        //---------------------------------------------------------------------
        // タッチ処理
        //---------------------------------------------------------------------

        // ● タッチ処理
        _class.prototype.processTouch = function () {
            if (this.isClickHandled() && this.isActive()) {
                if (TouchInput.isTriggered() && this.isButtonTouched()) {
                    this._touching = true;
                }
                if (this._touching) {
                    if (TouchInput.isReleased() || !this.isButtonTouched()) {
                        this._touching = false;
                        if (TouchInput.isReleased()) {
                            this.callClickHandler();
                        }
                    }
                }
            } else {
                this._touching = false;
            }
        };

        // ● アクティブ状態の判定
        _class.prototype.isActive = function () {
            var node = this;
            while (node) {
                if (!node.visible) {
                    return false;
                }
                node = node.parent;
            }
            return true;
        };

        // ● ボタンのタッチ判定
        _class.prototype.isButtonTouched = function () {
            var x = this.canvasToLocalX(TouchInput.x);
            var y = this.canvasToLocalY(TouchInput.y);
            return x >= 0 && y >= 0 && x < this.width && y < this.height;
        };

        //---------------------------------------------------------------------
        // 座標処理
        //---------------------------------------------------------------------

        // ● X 絶対座標の取得
        _class.prototype.absoluteX = function () {
            var x = 0;
            var object = this;
            while (object) {
                x += object.x;
                object = object.parent;
            }
            return x;
        };

        // ●  Y 絶対座標の取得
        _class.prototype.absoluteY = function () {
            var y = 0;
            var object = this;
            while (object) {
                y += object.y;
                object = object.parent;
            }
            return y;
        };

        // ● 絶対座標からローカルX座標に変換
        _class.prototype.canvasToLocalX = function (x) {
            return x - this.absoluteX();
        };

        // ● 絶対座標からローカルY座標に変換
        _class.prototype.canvasToLocalY = function (y) {
            return y - this.absoluteY();
        };

        //---------------------------------------------------------------------
        // フォント処理
        //---------------------------------------------------------------------

        // ● ライン高さの取得
        _class.prototype.lineHeight = function () {
            return Window_Base.prototype.lineHeight.call(this);
        };

        // ● フォントの取得
        _class.prototype.standardFontFace = function () {
            return Window_Base.prototype.standardFontFace.call(this);
        };

        // ● フォントサイズの取得
        _class.prototype.standardFontSize = function () {
            return Window_Base.prototype.standardFontSize.call(this);
        };

        // ● フォント設定のリセット
        _class.prototype.resetFontSettings = function () {
            this.contents.fontFace = this.standardFontFace();
            this.contents.fontSize = this.standardFontSize();
            this.resetTextColor();
            this.resetPaintOpacity();
        };

        // ● 文字色の変更
        _class.prototype.changeTextColor = function (color) {
            this.contents.textColor = color;
        };

        // ● 文字色のリセット
        _class.prototype.resetTextColor = function () {
            this.changeTextColor(this.normalColor());
        };

        // ● 通常文字色の取得
        _class.prototype.normalColor = function () {
            return _normalColor;
        };

        // ● システム文字色の取得
        _class.prototype.systemColor = function () {
            return _systemColor;
        };

        // ● ピンチ文字色の取得
        _class.prototype.crisisColor = function () {
            return _crisisColor;
        };

        // ● 戦闘不能文字色の取得
        _class.prototype.deathColor = function () {
            return _deathColor;
        };

        // ● 能力上昇文字色の取得
        _class.prototype.powerUpColor = function () {
            return _powerUpColor;
        };

        // ● 能力減少文字色の取得
        _class.prototype.powerDownColor = function () {
            return _powerDownColor;
        };

        // ● 消費MP文字色の取得
        _class.prototype.mpCostColor = function () {
            return _mpCostColor;
        };

        // ● 消費TP文字色の取得
        _class.prototype.tpCostColor = function () {
            return _tpCostColor;
        };

        // ● ゲージ背景色の取得
        _class.prototype.gaugeBackColor = function () {
            return _gaugeBackColor;
        };

        // ● HPゲージ色１の取得
        _class.prototype.hpGaugeColor1 = function () {
            return _hpGaugeColor1;
        };

        // ● HPゲージ色２の取得
        _class.prototype.hpGaugeColor2 = function () {
            return _hpGaugeColor2;
        };

        // ● MPゲージ色１の取得
        _class.prototype.mpGaugeColor1 = function () {
            return _mpGaugeColor1;
        };

        // ● MPゲージ色２の取得
        _class.prototype.mpGaugeColor2 = function () {
            return _mpGaugeColor2;
        };

        // ● TPゲージ色１の取得
        _class.prototype.tpGaugeColor1 = function () {
            return _tpGaugeColor1;
        };

        // ● TPゲージ色２の取得
        _class.prototype.tpGaugeColor2 = function () {
            return _tpGaugeColor2;
        };

        // ● HP文字色の取得
        _class.prototype.hpColor = function (actor) {
            if (actor.isDead()) {
                return this.deathColor();
            } else if (actor.isDying()) {
                return this.crisisColor();
            } else {
                return this.normalColor();
            }
        };

        // ● MP文字色の取得
        _class.prototype.mpColor = function (actor) {
            return this.normalColor();
        };

        // ● TP文字色の取得
        _class.prototype.tpColor = function (actor) {
            return this.normalColor();
        };

        // ● 描画透明度のリセット
        _class.prototype.resetPaintOpacity = function () {
            this.contents.paintOpacity = 255;
        };

        // ● 無効時の透明度の取得
        _class.prototype.translucentOpacity = function () {
            return Window_Base.prototype.translucentOpacity.call(this);
        };

        // ● 描画透明度の変更
        _class.prototype.changePaintOpacity = function (enabled) {
            this.contents.paintOpacity = enabled ? 255 : this.translucentOpacity();
        };

        //---------------------------------------------------------------------
        // 描画処理
        //---------------------------------------------------------------------

        // ● ビットマップの描画
        _class.prototype.blt = function (bitmap, px, py, pw, ph, x, y, width, height) {
            this.contents.blt(bitmap, px, py, pw, ph, x, y, width, height);
        };

        // ● スキンの描画
        _class.prototype.bltSkin = function (bitmap, m, px, py, pw, ph, x, y, w, h) {
            var pxm = px + m;
            var pym = py + m;
            var pwm = px + pw - m;
            var phm = py + ph - m;
            var pcw = pw - m * 2;
            var pch = ph - m * 2;
            var xm = x + m;
            var ym = y + m;
            var wm = x + w - m;
            var hm = y + h - m;
            var cw = w - m * 2;
            var ch = h - m * 2;
            this.contents.blt(bitmap, pxm, pym, pcw, pch, xm, ym, cw, ch);
            this.contents.blt(bitmap, pxm, py, pcw, m, xm, y, cw, m);
            this.contents.blt(bitmap, pxm, phm, pcw, m, xm, hm, cw, m);
            this.contents.blt(bitmap, px, pym, m, pch, x, ym, m, ch);
            this.contents.blt(bitmap, pwm, pym, m, pch, wm, ym, m, ch);
            this.contents.blt(bitmap, px, py, m, m, x, y, m, m);
            this.contents.blt(bitmap, pwm, py, m, m, wm, y, m, m);
            this.contents.blt(bitmap, px, phm, m, m, x, hm, m, m);
            this.contents.blt(bitmap, pwm, phm, m, m, wm, hm, m, m);
        };

        // ● テキストの描画
        _class.prototype.drawText = function (text, x, y, maxWidth, align) {
            this.contents.drawText(text, x, y, maxWidth, this.lineHeight(), align);
        };

        // ● テキスト幅の取得
        _class.prototype.textWidth = function (text) {
            return this.contents.measureTextWidth(text);
        };

        // ● アイコン付きテキストの描画
        _class.prototype.drawTextWithIcon = function (text, iconIndex, x, y, width) {
            var pw = Window_Base._iconWidth;
            var ph = Window_Base._iconHeight;
            this.drawIcon(iconIndex, x + 2, y + 2);
            this.contents.drawText(text, x + pw + 4, y, width - pw - 4, ph + 4);
        };

        // ● アイコンの描画
        _class.prototype.drawIcon = function (iconIndex, x, y) {
            var bitmap = ImageManager.loadSystem('IconSet');
            var pw = Window_Base._iconWidth;
            var ph = Window_Base._iconHeight;
            var sx = iconIndex % 16 * pw;
            var sy = Math.floor(iconIndex / 16) * ph;
            this.contents.blt(bitmap, sx, sy, pw, ph, x, y);
        };

        // ● リングの描画
        _class.prototype.drawIconRing = function (ringIndex, x, y) {
            var bitmap = ImageManager.loadSystem(_imgIconRing);
            var pw = _class._iconRingWidth;
            var ph = _class._iconRingHeight;
            var sx = ringIndex % 8 * pw;
            var sy = Math.floor(ringIndex / 8) * ph;
            this.contents.blt(bitmap, sx, sy, pw, ph, x, y);
        };

        // ● 顔グラフィックの描画
        _class.prototype.drawFace = function (faceName, faceIndex, x, y, width, height) {
            width = width || Window_Base._faceWidth;
            height = height || Window_Base._faceHeight;
            var bitmap = ImageManager.loadFace(faceName);
            var pw = Window_Base._faceWidth;
            var ph = Window_Base._faceHeight;
            var sw = Math.min(width, pw);
            var sh = Math.min(height, ph);
            var dx = Math.floor(x + Math.max(width - pw, 0) / 2);
            var dy = Math.floor(y + Math.max(height - ph, 0) / 2);
            var sx = faceIndex % 4 * pw + (pw - sw) / 2;
            var sy = Math.floor(faceIndex / 4) * ph + (ph - sh) / 2;
            this.contents.blt(bitmap, sx, sy, sw, sh, dx, dy);
        };

        // ● キャラクターの描画
        _class.prototype.drawCharacter = function (characterName, characterIndex, x, y) {
            var bitmap = ImageManager.loadCharacter(characterName);
            var big = ImageManager.isBigCharacter(characterName);
            var pw = bitmap.width / (big ? 3 : 12);
            var ph = bitmap.height / (big ? 4 : 8);
            var n = characterIndex;
            var sx = (n % 4 * 3 + 1) * pw;
            var sy = (Math.floor(n / 4) * 4) * ph;
            this.contents.blt(bitmap, sx, sy, pw, ph, x - pw / 2, y - ph);
        };

        // ● ゲージの描画
        _class.prototype.drawGauge = function (x, y, width, rate, color1, color2) {
            var fillW = Math.floor(width * rate);
            var gaugeY = y + this.lineHeight() - 8;
            this.contents.fillRect(x, gaugeY, width, 6, this.gaugeBackColor());
            this.contents.gradientFillRect(x, gaugeY, fillW, 6, color1, color2);
        };

        // ● 現在値と最大値の描画
        _class.prototype.drawCurrentAndMax = function (current, max, x, y,
            width, color1, color2) {
            var labelWidth = this.textWidth('HP');
            var valueWidth = this.textWidth('0000');
            var slashWidth = this.textWidth('/');
            var x1 = x + width - valueWidth;
            var x2 = x1 - slashWidth;
            var x3 = x2 - valueWidth;
            if (x3 >= x + labelWidth) {
                this.changeTextColor(color1);
                this.drawText(current, x3, y, valueWidth, 'right');
                this.changeTextColor(color2);
                this.drawText('/', x2, y, slashWidth, 'right');
                this.drawText(max, x1, y, valueWidth, 'right');
            } else {
                this.changeTextColor(color1);
                this.drawText(current, x1, y, valueWidth, 'right');
            }
        };

        // ● アクターのキャラクターの描画
        _class.prototype.drawActorCharacter = function (actor, x, y) {
            this.drawCharacter(actor.characterName(), actor.characterIndex(), x, y);
        };

        // ● アクターの顔グラフィックの描画
        _class.prototype.drawActorFace = function (actor, x, y, width, height) {
            this.drawFace(actor.faceName(), actor.faceIndex(), x, y, width, height);
        };

        // ● アクターの名前の描画
        _class.prototype.drawActorName = function (actor, x, y, width) {
            width = width || 168;
            this.changeTextColor(this.hpColor(actor));
            this.drawText(actor.name(), x, y, width);
        };

        // ● アクターの職業の描画
        _class.prototype.drawActorClass = function (actor, x, y, width) {
            width = width || 168;
            this.resetTextColor();
            this.drawText(actor.currentClass().name, x, y, width);
        };

        // ● アクターのニックネームの描画
        _class.prototype.drawActorNickname = function (actor, x, y, width) {
            width = width || 270;
            this.resetTextColor();
            this.drawText(actor.nickname(), x, y, width);
        };

        // ● アクターのレベルの描画
        _class.prototype.drawActorLevel = function (actor, x, y) {
            this.changeTextColor(this.systemColor());
            this.drawText(TextManager.levelA, x, y, 48);
            this.resetTextColor();
            this.drawText(actor.level, x + 84, y, 36, 'right');
        };

        // ● アクターのアイコンの描画
        _class.prototype.drawActorIcons = function (actor, x, y, width) {
            width = width || 144;
            var icons = actor.allIcons().slice(0, Math.floor(width / Window_Base._iconWidth));
            for (var i = 0; i < icons.length; i++) {
                this.drawIcon(icons[i], x + Window_Base._iconWidth * i, y + 2);
            }
        };

        // ● アクターのHPの描画
        _class.prototype.drawActorHp = function (actor, x, y, width) {
            width = width || 186;
            var color1 = this.hpGaugeColor1();
            var color2 = this.hpGaugeColor2();
            this.drawGauge(x, y, width, actor.hpRate(), color1, color2);
            this.changeTextColor(this.systemColor());
            this.drawText(TextManager.hpA, x, y, 44);
            this.drawCurrentAndMax(actor.hp, actor.mhp, x, y, width,
                this.hpColor(actor), this.normalColor());
        };

        // ● アクターのMPの描画
        _class.prototype.drawActorMp = function (actor, x, y, width) {
            width = width || 186;
            var color1 = this.mpGaugeColor1();
            var color2 = this.mpGaugeColor2();
            this.drawGauge(x, y, width, actor.mpRate(), color1, color2);
            this.changeTextColor(this.systemColor());
            this.drawText(TextManager.mpA, x, y, 44);
            this.drawCurrentAndMax(actor.mp, actor.mmp, x, y, width,
                this.mpColor(actor), this.normalColor());
        };

        // ● アクターのTPの描画
        _class.prototype.drawActorTp = function (actor, x, y, width) {
            width = width || 96;
            var color1 = this.tpGaugeColor1();
            var color2 = this.tpGaugeColor2();
            this.drawGauge(x, y, width, actor.tpRate(), color1, color2);
            this.changeTextColor(this.systemColor());
            this.drawText(TextManager.tpA, x, y, 44);
            this.changeTextColor(this.tpColor(actor));
            this.drawText(actor.tp, x + width - 64, y, 64, 'right');
        };

        // ● アイテム名の描画
        _class.prototype.drawItemName = function (item, x, y, width) {
            width = width || 312;
            if (item) {
                var iconBoxWidth = Window_Base._iconWidth + 4;
                this.resetTextColor();
                this.drawIcon(item.iconIndex, x + 2, y + 2);
                this.drawText(item.name, x + iconBoxWidth, y, width - iconBoxWidth);
            }
        };

        // ● 所持金の描画
        _class.prototype.drawCurrencyValue = function (value, unit, x, y, width) {
            var unitWidth = Math.min(80, this.textWidth(unit));
            this.resetTextColor();
            this.drawText(value, x, y, width - unitWidth - 6, 'right');
            this.changeTextColor(this.systemColor());
            this.drawText(unit, x + width - unitWidth, y, unitWidth, 'right');
        };

    })();

    //=========================================================================
    // Sprite_Data
    // 　リストやアイテムなどのデータを扱う描画スプライトのベースとなるクラスです。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Sprite_Data() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.Sprite_Drawing;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Sprite_Data = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function (x, y, width, height) {
            _super.prototype.initialize.call(this, x, y, width, height);
            this._type = '';
            this._data = null;
            this._battler = null;
        };

        // ● リフレッシュ
        _class.prototype.refresh = function () {
            this.contents.clear();
            this.drawContents();
        };

        //---------------------------------------------------------------------
        // 情報処理
        //---------------------------------------------------------------------

        // ● データの設定
        _class.prototype.setData = function (type, data, battler) {
            battler = battler || null;
            if (this._type !== type || this._data !== data || this._battler !== battler) {
                this._type = type;
                this._data = data;
                this._battler = battler;
                this.refresh();
            }
        };

        // ● リストの設定
        _class.prototype.setList = function (list, battler) {
            this.setData('list', list, battler);
        };

        // ● リストの設定
        _class.prototype.setItem = function (item, battler) {
            this.setData('item', item, battler);
        };

        // ● データのクリア
        _class.prototype.clear = function () {
            this.setData('', null, null);
        };

        // ● テキストの取得
        _class.prototype.text = function () {
            var data = this._data;
            switch (this._type) {
                case 'list':
                    return data.name;
                case 'item':
                    return data.name;
                case 'data':
                    return data.name;
            }
            return '';
        };

        // ● シンボルの取得
        _class.prototype.symbol = function () {
            var data = this._data;
            switch (this._type) {
                case 'list':
                    return data.symbol;
                case 'item':
                    return this.itemSymbol(data);
                case 'data':
                    return data.symbol;
            }
            return '';
        };

        // ● 有効判定
        _class.prototype.isEnabled = function () {
            var data = this._data;
            switch (this._type) {
                case 'list':
                    return data.enabled;
                case 'item':
                    return this.isItemEnabled(data);
                case 'data':
                    return data.enabled;
            }
            return false;
        };

        // ● アイテムのシンボルの取得
        _class.prototype.itemSymbol = function (item) {
            // メモ : 表示形式に合わせて再定義を行う。
            return '';
        };

        // ● アイテムの有効判定
        _class.prototype.isItemEnabled = function (item) {
            // メモ : 表示形式に合わせて再定義を行う。
            return !!item;
        };

        //---------------------------------------------------------------------
        // 描画処理
        //---------------------------------------------------------------------

        // ● 内容の描画 (継承先で再定義する)
        _class.prototype.drawContents = function () {};

    })();

    //=========================================================================
    // Sprite_Command
    // 　コマンド画像とコマンド名を描画するスプライトのクラスです。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Sprite_Command() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.Sprite_Data;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Sprite_Command = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function (x, y, width, height) {
            _super.prototype.initialize.call(this, x, y, width, height);
        };

        //---------------------------------------------------------------------
        // 描画処理
        //---------------------------------------------------------------------

        // ● 内容の描画
        _class.prototype.drawContents = function () {
            this.changePaintOpacity(this.isEnabled());
            this.drawCommandSkin();
            this.drawCommandText();
        };

        // ● コマンドスキンの描画
        _class.prototype.drawCommandSkin = function () {
            var bitmap = ImageManager.loadSystem(_imgCommandSkin);
            var pw = bitmap.width;
            var ph = bitmap.height;
            var width = this.width;
            var height = this.height;
            this.bltSkin(bitmap, 12, 0, 0, pw, ph, 0, 0, width, height);
        };

        // ● コマンド名の描画
        _class.prototype.drawCommandText = function () {
            var textY = (this.height - this.lineHeight()) / 2;
            this.drawText(this.text(), 8, textY, this.width - 16, 'center');
        };

    })();

    //=========================================================================
    // Sprite_IconRing
    // 　アイコンリング画像を描画するスプライトのクラスです。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Sprite_IconRing() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.Sprite_Data;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Sprite_IconRing = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function (x, y) {
            var width = $.Sprite_Drawing._iconRingWidth;
            var height = $.Sprite_Drawing._iconRingHeight;
            _super.prototype.initialize.call(this, x, y, width, height);
        };

        //---------------------------------------------------------------------
        // 情報処理
        //---------------------------------------------------------------------

        // ● アイコン番号の取得
        _class.prototype.iconIndex = function () {
            var data = this._data;
            switch (this._type) {
                case 'list':
                    return this.symbolIconIndex(data.symbol, data.ext);
                case 'item':
                    return data.iconIndex;
                case 'data':
                    return data.iconIndex;
            }
            return 0;
        };

        // ● リング番号の取得
        _class.prototype.ringIndex = function () {
            var data = this._data;
            switch (this._type) {
                case 'list':
                    return this.symbolRingIndex(data.symbol, data.ext);
                case 'item':
                    return data.meta.exRingIndex;
                case 'data':
                    return data.ringIndex;
            }
            return 1;
        };

        // ● シンボルのアイコン番号を取得
        _class.prototype.symbolIconIndex = function (symbol, id) {
            // メモ : 表示形式に合わせて再定義を行う。
            return 0;
        };

        // ● シンボルのリング番号を取得
        _class.prototype.symbolRingIndex = function (symbol, id) {
            // メモ : 表示形式に合わせて再定義を行う。
            return 1;
        };

        //---------------------------------------------------------------------
        // フォント処理
        //---------------------------------------------------------------------

        // ● ライン高さの取得
        _class.prototype.lineHeight = function () {
            return 24;
        };

        // ● フォントサイズの取得
        _class.prototype.standardFontSize = function () {
            return 20;
        };

        // ● フォントのリセット
        _class.prototype.resetFontSettings = function () {
            _super.prototype.resetFontSettings.call(this);
            this.contents.outlineColor = 'rgba(0, 0, 0, 0.8)';
            this.contents.outlineWidth = 3;
        };

        //---------------------------------------------------------------------
        // 描画処理
        //---------------------------------------------------------------------

        // ● 内容の描画
        _class.prototype.drawContents = function () {
            var ix = (this.width - Window_Base._iconWidth) / 2;
            var iy = (this.height - Window_Base._iconHeight) / 2;
            this.changePaintOpacity(this.isEnabled());
            this.drawIconRing(this.ringIndex(), 0, 0);
            this.drawIcon(this.iconIndex(), ix, iy);
        };

        // ● アイコンリング文字の描画
        _class.prototype.drawIconRingText = function (text, color) {
            var y = this.height - this.lineHeight();
            this.changeTextColor(color);
            this.drawText(text, 0, y, this.width, 'center');
        };

    })();

    //=========================================================================
    // LayerContainer
    // 　座標関連を定義したコンテナーです。Perch関連クラスで継承して使用します。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function LayerContainer() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = PIXI.Container || (function () {}); // 予測用おまじない

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.LayerContainer = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function () {
            _super.call(this);
            this._isLayerContainer = true;
            this._width = 0;
            this._height = 0;
            this._setupCanvasRendering();
            this._setupWebGLRendering();
        };

        // ● フレーム更新
        _class.prototype.update = function () {
            this.children.forEach(function (child) {
                if (child.update) {
                    child.update();
                }
            });
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● 最初の子オブジェクトを取り除く
        _class.prototype.shiftChild = function () {
            if (this.children.length > 0) {
                return this.removeChildAt(0);
            }
            return null;
        };

        // ● 最後の子オブジェクトを取り除く
        _class.prototype.popChild = function () {
            if (this.children.length > 0) {
                return this.removeChildAt(this.children.length - 1);
            }
            return null;
        };

        // ● レイヤーのリフレッシュ
        _class.prototype.refreshLayer = function () {};

        //---------------------------------------------------------------------
        // タッチ処理
        //---------------------------------------------------------------------

        // ● タッチが範囲内で行われたか判定
        _class.prototype.isTouchedInsideFrame = function () {
            var x = this.canvasToLocalX(TouchInput.x);
            var y = this.canvasToLocalY(TouchInput.y);
            return x >= 0 && y >= 0 && x < this.width && y < this.height;
        };

        //---------------------------------------------------------------------
        // 座標処理
        //---------------------------------------------------------------------

        // ● レイヤー幅
        Object.defineProperty(_class.prototype, 'width', {
            get: function () {
                return this._width;
            },
            set: function (value) {
                this._width = value;
                this.refreshLayer();
            },
            configurable: true
        });

        // ● レイヤー高さ
        Object.defineProperty(_class.prototype, 'height', {
            get: function () {
                return this._height;
            },
            set: function (value) {
                this._height = value;
                this.refreshLayer();
            },
            configurable: true
        });

        // ● 移動する
        _class.prototype.move = function (x, y, width, height) {
            this.x = x || 0;
            this.y = y || 0;
            if (this._width !== width || this._height !== height) {
                this._width = width || 0;
                this._height = height || 0;
                this.refreshLayer();
            }
        };

        // ● X 絶対座標の取得
        _class.prototype.absoluteX = function () {
            var x = 0;
            var object = this;
            while (object) {
                x += object.x;
                object = object.parent;
            }
            return x;
        };

        // ●  Y 絶対座標の取得
        _class.prototype.absoluteY = function () {
            var y = 0;
            var object = this;
            while (object) {
                y += object.y;
                object = object.parent;
            }
            return y;
        };

        // ● 絶対座標からローカルX座標に変換
        _class.prototype.canvasToLocalX = function (x) {
            return x - this.absoluteX();
        };

        // ● 絶対座標からローカルY座標に変換
        _class.prototype.canvasToLocalY = function (y) {
            return y - this.absoluteY();
        };

        // ● 対象からローカルX座標に変換
        _class.prototype.targetToLocalX = function (target) {
            var x = 0;
            var object = target;
            while (object !== this) {
                if (!object) {
                    return x - this.absoluteX();
                }
                x += object.x;
                object = object.parent;
            }
            return x;
        };

        // ● 対象からローカルY座標に変換
        _class.prototype.targetToLocalY = function (target) {
            var y = 0;
            var object = target;
            while (object !== this) {
                if (!object) {
                    return y - this.absoluteY();
                }
                y += object.y;
                object = object.parent;
            }
            return y;
        };

        // ● 対象のX座標の取得
        _class.prototype.targetX = function (target) {
            return this.targetToLocalX(target);
        };

        // ● 対象のY座標の取得
        _class.prototype.targetY = function (target) {
            return this.targetToLocalY(target);
        };

        // ● 対象の幅の取得
        _class.prototype.targetWidth = function (target) {
            return target.width;
        };

        // ● 対象の高さの取得
        _class.prototype.targetHeight = function (target) {
            return target.height;
        };

        // ● 対象の領域の取得
        _class.prototype.targetRect = function (target) {
            if (target) {
                var x = this.targetX(target);
                var y = this.targetY(target);
                var width = this.targetWidth(target);
                var height = this.targetHeight(target);
                return new Rectangle(x, y, width, height);
            }
            return Rectangle.emptyRectangle;
        };

        //---------------------------------------------------------------------
        // Canvas描画
        //---------------------------------------------------------------------

        // ● Canvas描画のセットアップ
        _class.prototype._setupCanvasRendering = function () {};

        //---------------------------------------------------------------------
        // WebGL描画
        //---------------------------------------------------------------------

        // ● WebGL描画のセットアップ
        _class.prototype._setupWebGLRendering = function () {};

    })();

    //=========================================================================
    // EXLayer_RenderingMask
    // 　表示領域を超える範囲をマスクします。
    //=========================================================================

    $.EXLayer_RenderingMask = function (_class, _super) {

        //---------------------------------------------------------------------
        // Canvas描画
        //---------------------------------------------------------------------

        // ● Canvas描画のセットアップ
        _class.prototype._setupCanvasRendering = function () {
            this._tempCanvas = null;
        };

        // ● Canvas描画処理
        _class.prototype.renderCanvas = function (renderer) {
            if (!this.visible || !this.renderable) {
                return;
            }

            if (!this._tempCanvas) {
                this._tempCanvas = document.createElement('canvas');
            }

            this._tempCanvas.width = Graphics.width;
            this._tempCanvas.height = Graphics.height;

            var realCanvasContext = renderer.context;
            var context = this._tempCanvas.getContext('2d');

            context.save();
            context.clearRect(0, 0, Graphics.width, Graphics.height);
            context.beginPath();
            context.rect(this.absoluteX(), this.absoluteY(), this.width, this.height);
            context.closePath();
            context.clip();

            renderer.context = context;

            for (var i = 0; i < this.children.length; i++) {
                this.children[i].renderCanvas(renderer);
            }

            context.restore();

            renderer.context = realCanvasContext;
            renderer.context.setTransform(1, 0, 0, 1, 0, 0);
            renderer.context.globalCompositeOperation = 'source-over';
            renderer.context.globalAlpha = 1;
            renderer.context.drawImage(this._tempCanvas, 0, 0);
        };

        //---------------------------------------------------------------------
        // WebGL描画
        //---------------------------------------------------------------------

        // ● WebGL描画のセットアップ
        _class.prototype._setupWebGLRendering = function () {
            this.filterArea = new PIXI.Rectangle();
            this.filters = [_class.voidFilter];
        };

        // ● WebGL描画用フィルターの設定
        _class.voidFilter = new PIXI.filters.VoidFilter();

        // ● WebGL描画処理
        _class.prototype.renderWebGL = function (renderer) {
            if (!this.visible || !this.renderable) {
                return;
            }

            if (this.children.length === 0) {
                return;
            }

            renderer.flush();
            this.filterArea.x = this.absoluteX();
            this.filterArea.y = this.absoluteY();
            this.filterArea.width = this.width;
            this.filterArea.height = this.height;
            renderer.filterManager.pushFilter(this, this.filters);
            renderer.currentRenderer.start();

            for (var i = 0; i < this.children.length; i++) {
                this.children[i].renderWebGL(renderer);
            }

            renderer.flush();
            renderer.filterManager.popFilter();
            renderer.currentRenderer.start();
        };

    };

    //=========================================================================
    // Perch
    // 　画像をまとめるオブジェクトのコアクラスです。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Perch() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.LayerContainer;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Perch = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function () {
            _super.prototype.initialize.call(this);

            this._isWindow = false;
            this._isPerch = true;
            this._cursorRect = new Rectangle();
            this._openness = 255;
            this._animationCount = 0;

            this._contentsBitmap = null;

            this._cursorPosition = 'top-right';
            this._cursorOffsetX = 0;
            this._cursorOffsetY = 0;
            this._cursorInterval = 10;
            this._cursorDisplay = 'front';
            this._cursorSprite = null;
            this._cursorBitmap = null;

            this._createAllParts();

            this.active = false;

            this.cursorVisible = false;
        };

        // ● フレーム更新
        _class.prototype.update = function () {
            if (this.active) {
                this._animationCount++;
            }
            _super.prototype.update.call(this);
        };

        // ● 状態の更新
        _class.prototype.updateTransform = function () {
            this._updateCursor();
            _super.prototype.updateTransform.call(this);
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● パーツの作成
        _class.prototype._createAllParts = function () {
            this._createCursor();
        };

        // ● 子オブジェクトを追加
        _class.prototype.addChild = function (child) {
            // メモ : 通常カーソル位置の前後に追加します。
            var index = this.children.indexOf(this._cursorSprite);
            switch (this._cursorDisplay) {
                case 'front':
                    return this.addChildAt(child, index);
                case 'behind':
                    return this.addChildAt(child, index + 1);
            }
            return this.addChildAt(child, this.children.length);
        };

        // ● 前面に子オブジェクトを追加
        _class.prototype.addChildToFront = function (child) {
            return this.addChildAt(child, this.children.length);
        };

        // ● 背面に子オブジェクトを追加
        _class.prototype.addChildToBack = function (child) {
            return this.addChildAt(child, 0);
        };

        //---------------------------------------------------------------------
        // 状態処理
        //---------------------------------------------------------------------

        // ● オープン度
        Object.defineProperty(_class.prototype, 'openness', {
            get: function () {
                return this._openness;
            },
            set: function (value) {
                if (this._openness !== value) {
                    this._openness = value.clamp(0, 255);
                }
            },
            configurable: true
        });

        // ● オープン状態判定
        _class.prototype.isOpen = function () {
            return this._openness >= 255;
        };

        // ● クローズ状態判定
        _class.prototype.isClosed = function () {
            return this._openness <= 0;
        };

        //---------------------------------------------------------------------
        // カーソル処理
        //---------------------------------------------------------------------

        // ● カーソルビットマップ
        Object.defineProperty(_class.prototype, 'cursorBitmap', {
            get: function () {
                return this._cursorBitmap;
            },
            set: function (value) {
                if (this._cursorBitmap !== value) {
                    this._cursorBitmap = value;
                    this._cursorBitmap.addLoadListener(this._onCursorBitmapLoad.bind(this));
                }
            },
            configurable: true
        });

        // ● カーソル位置
        Object.defineProperty(_class.prototype, 'cursorPosition', {
            get: function () {
                return this._cursorPosition;
            },
            set: function (value) {
                if (this._cursorPosition !== value) {
                    this._cursorPosition = value;
                    this._refreshCursor();
                }
            },
            configurable: true
        });

        // ● カーソルオフセットX座標
        Object.defineProperty(_class.prototype, 'cursorOffsetX', {
            get: function () {
                return this._cursorOffsetX;
            },
            set: function (value) {
                if (this._cursorOffsetX !== value) {
                    this._cursorOffsetX = value;
                    this._refreshCursor();
                }
            },
            configurable: true
        });

        // ● カーソルオフセットY座標
        Object.defineProperty(_class.prototype, 'cursorOffsetY', {
            get: function () {
                return this._cursorOffsetY;
            },
            set: function (value) {
                if (this._cursorOffsetY !== value) {
                    this._cursorOffsetY = value;
                    this._refreshCursor();
                }
            },
            configurable: true
        });

        // ● カーソル更新間隔
        Object.defineProperty(_class.prototype, 'cursorInterval', {
            get: function () {
                return this._cursorInterval;
            },
            set: function (value) {
                this._cursorInterval = value;
            },
            configurable: true
        });

        // ● カーソルの配置
        Object.defineProperty(_class.prototype, 'cursorDisplay', {
            get: function () {
                return this._cursorDisplay;
            },
            set: function (value) {
                this._cursorDisplay = value;
            },
            configurable: true
        });

        // ● カーソルの表示
        Object.defineProperty(_class.prototype, 'cursorVisible', {
            get: function () {
                return this._cursorSprite.visible;
            },
            set: function (value) {
                this._cursorSprite.visible = value;
            },
            configurable: true
        });

        // ● カーソルビットマップのロード
        _class.prototype._onCursorBitmapLoad = function () {
            this._refreshCursor();
        };

        // ● カーソルの作成
        _class.prototype._createCursor = function () {
            this._cursorSprite = new Sprite();
            this._cursorSprite.anchor.x = 0.5;
            this._cursorSprite.anchor.y = 0.5;
            this.addChildToFront(this._cursorSprite);
        };

        // ● カーソルのリフレッシュ
        _class.prototype._refreshCursor = function () {
            this._cursorSprite.bitmap = this._cursorBitmap;
            this._updateCursor();
            this._moveCursor();
        };

        // ● カーソルの更新
        _class.prototype._updateCursor = function () {
            if (this._cursorBitmap) {
                var size = this._cursorBitmap.height;
                if (this.active) {
                    var fs = this._cursorBitmap.width / size;
                    var fi = this._cursorInterval;
                    var px = size * Math.floor(this._animationCount / fi % fs);
                    this._cursorSprite.setFrame(px, 0, size, size);
                } else {
                    this._cursorSprite.setFrame(0, 0, size, size);
                }
            }
        };

        // ● カーソルの移動
        _class.prototype._moveCursor = function () {
            var rect = this._cursorRect;
            var cx = rect.x + rect.width / 2;
            var cy = rect.y + rect.height / 2;
            switch (this._cursorPosition) {
                case 'top':
                    cx = rect.x + rect.width / 2;
                    cy = rect.y;
                    break;
                case 'bottom':
                    cx = rect.x + rect.width / 2;
                    cy = rect.y + rect.height;
                    break;
                case 'left':
                    cx = rect.x;
                    cy = rect.y + rect.height / 2;
                    break;
                case 'right':
                    cx = rect.x + rect.width;
                    cy = rect.y + rect.height / 2;
                    break;
                case 'top-left':
                    cx = rect.x;
                    cy = rect.y;
                    break;
                case 'top-right':
                    cx = rect.x + rect.width;
                    cy = rect.y;
                    break;
                case 'bottom-left':
                    cx = rect.x;
                    cy = rect.y + rect.height;
                    break;
                case 'bottom-right':
                    cx = rect.x + rect.width;
                    cy = rect.y + rect.height;
                    break;
            }
            this._cursorSprite.x = cx + this._cursorOffsetX;
            this._cursorSprite.y = cy + this._cursorOffsetY;
        };

        // ● カーソル位置を設定
        _class.prototype.setCursorRect = function (x, y, width, height) {
            var cx = Math.floor(x || 0);
            var cy = Math.floor(y || 0);
            var cw = Math.floor(width || 0);
            var ch = Math.floor(height || 0);
            var rect = this._cursorRect;
            if (rect.x !== cx || rect.y !== cy || rect.width !== cw || rect.height !== ch) {
                this._cursorRect.x = cx;
                this._cursorRect.y = cy;
                this._cursorRect.width = cw;
                this._cursorRect.height = ch;
                this._refreshCursor();
            }
        };

    })();

    //=========================================================================
    // Perch_Base
    // 　画像をまとめるオブジェクトの基礎クラスです。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Perch_Base() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.Perch;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Perch_Base = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function (x, y, width, height) {
            _super.prototype.initialize.call(this);
            this.loadCursorBitmap();
            this.move(x, y, width, height);
            this._opening = false;
            this._closing = false;
            this._handlers = {};
            this._helpWindow = null;
        };

        // ● フレーム更新
        _class.prototype.update = function () {
            _super.prototype.update.call(this);
            this.updateOpen();
            this.updateClose();
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● カーソルビットマップのロード
        _class.prototype.loadCursorBitmap = function () {
            this.cursorBitmap = ImageManager.loadSystem(_imgPerchCursor);
        };

        // ● 顔グラフィックの準備
        _class.prototype.reserveFaceImages = function () {
            $gameParty.members().forEach(function (actor) {
                ImageManager.reserveFace(actor.faceName());
            }, this);
        };

        //---------------------------------------------------------------------
        // 状態処理
        //---------------------------------------------------------------------

        // ● オープン速度の取得
        _class.prototype.openingSpeed = function () {
            return 32;
        };

        // ● クローズ速度の取得
        _class.prototype.closingSpeed = function () {
            return 32;
        };

        // ● 動作状態の判定
        _class.prototype.isOpenAndActive = function () {
            return this.isOpen() && this.active;
        };

        // ● オープン中判定
        _class.prototype.isOpening = function () {
            return this._opening;
        };

        // ● クローズ中判定
        _class.prototype.isClosing = function () {
            return this._closing;
        };

        // ● アクティブ状態にする
        _class.prototype.activate = function () {
            this.active = true;
        };

        // ● アクティブ状態を解除する
        _class.prototype.deactivate = function () {
            this.active = false;
        };

        // ● 表示する
        _class.prototype.show = function () {
            this.visible = true;
        };

        // ● 隠す
        _class.prototype.hide = function () {
            this.visible = false;
        };

        // ● 開く
        _class.prototype.open = function () {
            if (!this.isOpen()) {
                this._opening = true;
            }
            this._closing = false;
        };

        // ● 閉じる
        _class.prototype.close = function () {
            if (!this.isClosed()) {
                this._closing = true;
            }
            this._opening = false;
        };

        // ● オープン更新
        _class.prototype.updateOpen = function () {
            if (this._opening) {
                this.openness += this.openingSpeed();
                if (this.isOpen()) {
                    this._opening = false;
                }
            }
        };

        // ● クローズ更新
        _class.prototype.updateClose = function () {
            if (this._closing) {
                this.openness -= this.closingSpeed();
                if (this.isClosed()) {
                    this._closing = false;
                }
            }
        };

        //---------------------------------------------------------------------
        // 情報処理
        //---------------------------------------------------------------------

        // ● ヘルプウィンドウの関連付け
        _class.prototype.setHelpWindow = function (helpWindow) {
            this._helpWindow = helpWindow;
            this.callUpdateHelp();
        };

        // ● ヘルプウィンドウを表示
        _class.prototype.showHelpWindow = function () {
            if (this._helpWindow) {
                this._helpWindow.show();
            }
        };

        // ● ヘルプウィンドウを隠す
        _class.prototype.hideHelpWindow = function () {
            if (this._helpWindow) {
                this._helpWindow.hide();
            }
        };

        // ● ヘルプウィンドウの更新を呼び出す
        _class.prototype.callUpdateHelp = function () {
            if (this.active && this._helpWindow) {
                this.updateHelp();
            }
        };

        // ● ヘルプウィンドウを更新
        _class.prototype.updateHelp = function () {
            this._helpWindow.clear();
        };

        // ● ヘルプウィンドウにアイテムを設定
        _class.prototype.setHelpWindowItem = function (item) {
            if (this._helpWindow) {
                this._helpWindow.setItem(item);
            }
        };

        //---------------------------------------------------------------------
        // ハンドラ処理
        //---------------------------------------------------------------------

        // ● ハンドラの設定
        _class.prototype.setHandler = function (symbol, method) {
            this._handlers[symbol] = method;
        };

        // ● ハンドラの存在判定
        _class.prototype.isHandled = function (symbol) {
            return !!this._handlers[symbol];
        };

        // ● ハンドラの呼び出し
        _class.prototype.callHandler = function (symbol) {
            if (this.isHandled(symbol)) {
                this._handlers[symbol]();
            }
        };

    })();

    //=========================================================================
    // Perch_Selection
    // 　画像選択の基礎クラスです。画像やウィンドウを追加して選択します。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Perch_Selection() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.Perch_Base;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Perch_Selection = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function (x, y, width, height) {
            _super.prototype.initialize.call(this, x, y, width, height);
            this._selector = this.noneSelector();
            this._cursorFixed = false;
            this._cursorAll = false;
            this._touching = false;
        };

        // ● フレーム更新
        _class.prototype.update = function () {
            _super.prototype.update.call(this);
            this.updateProcess();
        };

        // ● 操作処理の更新
        _class.prototype.updateProcess = function () {
            this.processCursorMove();
            this.processHandling();
            this.processWheel();
            this.processTouch();
        };

        // ● 位置の更新
        _class.prototype.updatePosition = function () {
            // メモ : 継承先によっては毎フレーム呼び出しが行われます。
            this.updateCursor();
        };

        // ● 入力情報の更新
        _class.prototype.updateInputData = function () {
            Input.update();
            TouchInput.update();
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● コンテンツレイヤーの取得
        _class.prototype.contentsLayer = function () {
            // メモ : 画像を挿入するレイヤーを設定する場合はここを変更する。
            return this;
        };

        //---------------------------------------------------------------------
        // 状態処理
        //---------------------------------------------------------------------

        // ● アクティブ状態にする
        _class.prototype.activate = function () {
            _super.prototype.activate.call(this);
            this.reselect();
        };

        // ● アクティブ状態を解除する
        _class.prototype.deactivate = function () {
            _super.prototype.deactivate.call(this);
            this.reselect();
        };

        //---------------------------------------------------------------------
        // 項目処理
        //---------------------------------------------------------------------

        // メモ : 基本はセレクタはインデックスとして扱います。

        // ● 要素数の取得 (継承先で再定義する)
        _class.prototype.maxItems = function () {
            // メモ : 基本はカーソル移動する項目の数。
            return 0;
        };

        // ● 選択可能なオブジェクト配列を取得 (継承先で再定義する)
        _class.prototype.selectableObjects = function () {
            return [];
        };

        // ● 選択中のオブジェクトの取得
        _class.prototype.selectedObject = function () {
            // メモ : インデックス形式以外の場合は再定義する。
            return this.selectableObjects()[this.selector()];
        };

        // ● 非選択セレクタの取得
        _class.prototype.noneSelector = function () {
            // メモ : インデックス形式以外の場合は再定義する。
            return -1;
        };

        // ● 対象のセレクタの取得
        _class.prototype.targetSelector = function (object) {
            // メモ : インデックス形式以外の場合は再定義する。
            var objects = this.selectableObjects();
            for (var i = 0; i < objects.length; i++) {
                if (objects[i] === object) {
                    return i;
                }
            }
            return this.noneSelector();
        };

        // ● セレクタの取得
        _class.prototype.selector = function () {
            return this._selector;
        };

        // ● セレクタの有効判定
        _class.prototype.isValidSelector = function (selector) {
            return selector !== this.noneSelector();
        };

        // ● 選択中の判定
        _class.prototype.isSelected = function () {
            return this.isValidSelector(this.selector());
        };

        // ● 項目の選択
        _class.prototype.select = function (selector) {
            this._selector = selector;
            this.ensureCursorVisible();
            this.updatePosition();
            this.callUpdateHelp();
        };

        // ● 項目の選択解除
        _class.prototype.deselect = function () {
            this.select(this.noneSelector());
        };

        // ● 項目の再選択
        _class.prototype.reselect = function () {
            this.select(this._selector);
        };

        //---------------------------------------------------------------------
        // カーソル処理
        //---------------------------------------------------------------------

        // ● カーソル移動処理
        _class.prototype.processCursorMove = function () {
            if (this.isCursorMovable()) {
                var lastSelector = this.selector();
                if (Input.isRepeated('down')) {
                    this.cursorDown(Input.isTriggered('down'));
                }
                if (Input.isRepeated('up')) {
                    this.cursorUp(Input.isTriggered('up'));
                }
                if (Input.isRepeated('right')) {
                    this.cursorRight(Input.isTriggered('right'));
                }
                if (Input.isRepeated('left')) {
                    this.cursorLeft(Input.isTriggered('left'));
                }
                if (!this.isHandled('pagedown') && Input.isTriggered('pagedown')) {
                    this.cursorPagedown();
                }
                if (!this.isHandled('pageup') && Input.isTriggered('pageup')) {
                    this.cursorPageup();
                }
                if (this.selector() !== lastSelector) {
                    this.playCursorSound();
                }
            }
        };

        // ● カーソル移動可能判定
        _class.prototype.isCursorMovable = function () {
            return (this.isOpenAndActive() && !this._cursorFixed &&
                !this._cursorAll && this.maxItems() > 0);
        };

        // ● カーソル固定状態の取得
        _class.prototype.cursorFixed = function () {
            return this._cursorFixed;
        };

        // ● カーソル固定状態の設定
        _class.prototype.setCursorFixed = function (cursorFixed) {
            this._cursorFixed = cursorFixed;
        };

        // ● カーソル全選択状態の取得
        _class.prototype.cursorAll = function () {
            return this._cursorAll;
        };

        // ● カーソル全選択状態の設定
        _class.prototype.setCursorAll = function (cursorAll) {
            this._cursorAll = cursorAll;
        };

        // ● カーソルを下に移動
        _class.prototype.cursorDown = function (wrap) {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● カーソルを上に移動
        _class.prototype.cursorUp = function (wrap) {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● カーソルを右に移動
        _class.prototype.cursorRight = function (wrap) {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● カーソルを左に移動
        _class.prototype.cursorLeft = function (wrap) {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● カーソルを下にページ移動
        _class.prototype.cursorPagedown = function () {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● カーソルを上にページ移動
        _class.prototype.cursorPageup = function () {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● カーソルが見える位置に移動
        _class.prototype.ensureCursorVisible = function () {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● カーソルの更新
        _class.prototype.updateCursor = function () {
            var object = this.selectedObject();
            if (object) {
                var x = this.targetX(object);
                var y = this.targetY(object);
                var width = this.targetWidth(object);
                var height = this.targetHeight(object);
                this.setCursorRect(x, y, width, height);
                this.cursorVisible = object.visible;
            } else {
                this.cursorVisible = false;
            }
        };

        //---------------------------------------------------------------------
        // タッチ処理
        //---------------------------------------------------------------------

        // ● タッチ処理
        _class.prototype.processTouch = function () {
            if (this.isTouchable()) {
                if (TouchInput.isTriggered() && this.isTouchedInsideFrame()) {
                    this.startTouch();
                    this.onTouch(true);
                } else if (TouchInput.isCancelled()) {
                    if (this.isCancelEnabled()) {
                        this.processCancel();
                    }
                }
                if (this._touching) {
                    if (TouchInput.isPressed()) {
                        this.onTouch(false);
                    } else {
                        this.endTouch();
                        this.nonTouch();
                    }
                }
            } else {
                this.nonTouch();
            }
        };

        // ● タッチ操作可能判定
        _class.prototype.isTouchable = function () {
            // メモ : 表示形式に合わせて再定義を行う。
            return false;
        };

        // ● タッチ開始処理
        _class.prototype.startTouch = function () {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● タッチ処理
        _class.prototype.onTouch = function (triggered) {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● タッチ終了処理
        _class.prototype.endTouch = function () {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● タッチしていない処理
        _class.prototype.nonTouch = function () {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        // ● ヒットテスト
        _class.prototype.hitTest = function (x, y) {
            if (this.isContentsArea(x, y)) {
                var target = null;
                var objects = this.selectableObjects();
                for (var i = 0; i < objects.length; i++) {
                    target = objects[i];
                    if (this.checkHitTarget(target, x, y)) {
                        return this.targetSelector(target);
                    }
                }
            }
            return this.noneSelector();
        };

        // ● コンテンツ範囲内の判定
        _class.prototype.isContentsArea = function (x, y) {
            return this.checkHitTarget(this.contentsLayer(), x, y);
        };

        // ● 対象へのヒット判定
        _class.prototype.checkHitTarget = function (target, x, y) {
            var left = this.targetX(target);
            var top = this.targetY(target);
            var right = left + this.targetWidth(target);
            var bottom = top + this.targetHeight(target);
            return (x >= left && y >= top && x < right && y < bottom);
        };

        // ● タッチ決定可能判定
        _class.prototype.isTouchOkEnabled = function () {
            return this.isOkEnabled();
        };

        //---------------------------------------------------------------------
        // ホイール処理
        //---------------------------------------------------------------------

        // ● ホイール処理
        _class.prototype.processWheel = function () {
            // メモ : 表示形式に合わせて再定義を行う。
        };

        //---------------------------------------------------------------------
        // ハンドラ処理
        //---------------------------------------------------------------------

        // ● ハンドラ処理
        _class.prototype.processHandling = function () {
            if (this.isOpenAndActive()) {
                if (this.isOkEnabled() && this.isOkTriggered()) {
                    this.processOk();
                } else if (this.isCancelEnabled() && this.isCancelTriggered()) {
                    this.processCancel();
                } else if (this.isHandled('pagedown') && Input.isTriggered('pagedown')) {
                    this.processPagedown();
                } else if (this.isHandled('pageup') && Input.isTriggered('pageup')) {
                    this.processPageup();
                }
            }
        };

        // ● 決定処理
        _class.prototype.processOk = function () {
            if (this.isCurrentItemEnabled()) {
                this.playOkSound();
                this.updateInputData();
                this.deactivate();
                this.callOkHandler();
            } else {
                this.playBuzzerSound();
            }
        };

        // ● 決定可能判定
        _class.prototype.isOkEnabled = function () {
            return this.isHandled('ok');
        };

        // ● 決定キーの押下判定
        _class.prototype.isOkTriggered = function () {
            return Input.isRepeated('ok');
        };

        // ● 現在の項目の選択可能判定
        _class.prototype.isCurrentItemEnabled = function () {
            return true;
        };

        // ● 決定処理を呼び出す
        _class.prototype.callOkHandler = function () {
            this.callHandler('ok');
        };

        // ● キャンセル処理
        _class.prototype.processCancel = function () {
            this.playCancelSound();
            this.updateInputData();
            this.deactivate();
            this.callCancelHandler();
        };

        // ● キャンセル可能判定
        _class.prototype.isCancelEnabled = function () {
            return this.isHandled('cancel');
        };

        // ● キャンセルキーの押下判定
        _class.prototype.isCancelTriggered = function () {
            return Input.isRepeated('cancel');
        };

        // ● キャンセル処理を呼び出す
        _class.prototype.callCancelHandler = function () {
            this.callHandler('cancel');
        };

        // ● ページアップ処理
        _class.prototype.processPageup = function () {
            this.playPageupSound();
            this.updateInputData();
            this.deactivate();
            this.callHandler('pageup');
        };

        // ● ページダウン処理
        _class.prototype.processPagedown = function () {
            this.playPagedownSound();
            this.updateInputData();
            this.deactivate();
            this.callHandler('pagedown');
        };

        //---------------------------------------------------------------------
        // サウンド処理
        //---------------------------------------------------------------------

        // ● カーソル音を鳴らす
        _class.prototype.playCursorSound = function () {
            SoundManager.playCursor();
        };

        // ● タッチ音を鳴らす
        _class.prototype.playTouchSound = function () {
            SoundManager.playCursor();
        };

        // ● 決定音を鳴らす
        _class.prototype.playOkSound = function () {
            SoundManager.playOk();
        };

        // ● キャンセル音を鳴らす
        _class.prototype.playCancelSound = function () {
            SoundManager.playCancel();
        };

        // ● ページアップ音を鳴らす
        _class.prototype.playPageupSound = function () {
            SoundManager.playCursor();
        };

        // ● ページダウン音を鳴らす
        _class.prototype.playPagedownSound = function () {
            SoundManager.playCursor();
        };

        // ● ブザー音を鳴らす
        _class.prototype.playBuzzerSound = function () {
            SoundManager.playBuzzer();
        };

    })();

    //=========================================================================
    // EXPerch_DistanceSelection
    // 　画像選択に、距離によるカーソル移動機能を追加します。
    //=========================================================================

    $.EXPerch_DistanceSelection = function (_class, _super) {

        // ● カーソルの指向性の取得
        _class.prototype.cursorDirectionality = function () {
            return 3;
        };

        // ● カーソルの移動
        _class.prototype.cursorMove = function (calcDistance, wrap) {
            var selector = this.selector();
            var current = this.selectedObject();
            if (current) {
                var distance = Number.MAX_VALUE;
                this.selectableObjects().forEach(function (target) {
                    var d = calcDistance(target, current);
                    if (d < distance) {
                        selector = this.targetSelector(target);
                        distance = d;
                    }
                }, this);
                this.select(selector);
            }
        };

        // ● カーソルを下に移動
        _class.prototype.cursorDown = function (wrap) {
            this.cursorMove(function (target, current) {
                var tx = this.targetX(target) + this.targetWidth(target) / 2;
                var ty = this.targetY(target) + this.targetHeight(target) / 2;
                var cx = this.targetX(current) + this.targetWidth(current) / 2;
                var cy = this.targetY(current) + this.targetHeight(current) / 2;
                if (ty > cy) {
                    var dx = (cx - tx) * this.cursorDirectionality();
                    var dy = cy - ty;
                    return Math.sqrt(dx * dx + dy * dy);
                } else {
                    return Infinity;
                }
            }.bind(this), wrap);
        };

        // ● カーソルを上に移動
        _class.prototype.cursorUp = function (wrap) {
            this.cursorMove(function (target, current) {
                var tx = this.targetX(target) + this.targetWidth(target) / 2;
                var ty = this.targetY(target) + this.targetHeight(target) / 2;
                var cx = this.targetX(current) + this.targetWidth(current) / 2;
                var cy = this.targetY(current) + this.targetHeight(current) / 2;
                if (ty < cy) {
                    var dx = (cx - tx) * this.cursorDirectionality();
                    var dy = cy - ty;
                    return Math.sqrt(dx * dx + dy * dy);
                } else {
                    return Infinity;
                }
            }.bind(this), wrap);
        };

        // ● カーソルを右に移動
        _class.prototype.cursorRight = function (wrap) {
            this.cursorMove(function (target, current) {
                var tx = this.targetX(target) + this.targetWidth(target) / 2;
                var ty = this.targetY(target) + this.targetHeight(target) / 2;
                var cx = this.targetX(current) + this.targetWidth(current) / 2;
                var cy = this.targetY(current) + this.targetHeight(current) / 2;
                if (tx > cx) {
                    var dx = cx - tx;
                    var dy = (cy - ty) * this.cursorDirectionality();
                    return Math.sqrt(dx * dx + dy * dy);
                } else {
                    return Infinity;
                }
            }.bind(this), wrap);
        };

        // ● カーソルを左に移動
        _class.prototype.cursorLeft = function (wrap) {
            this.cursorMove(function (target, current) {
                var tx = this.targetX(target) + this.targetWidth(target) / 2;
                var ty = this.targetY(target) + this.targetHeight(target) / 2;
                var cx = this.targetX(current) + this.targetWidth(current) / 2;
                var cy = this.targetY(current) + this.targetHeight(current) / 2;
                if (tx < cx) {
                    var dx = cx - tx;
                    var dy = (cy - ty) * this.cursorDirectionality();
                    return Math.sqrt(dx * dx + dy * dy);
                } else {
                    return Infinity;
                }
            }.bind(this), wrap);
        };

    };

    //=========================================================================
    // EXPerch_TouchSelection
    // 　画像選択に、タッチ操作で画像を選択する機能を追加します。
    //=========================================================================

    $.EXPerch_TouchSelection = function (_class, _super) {

        // ● タッチ操作可能判定
        _class.prototype.isTouchable = function () {
            return this.isOpenAndActive();
        };

        // ● タッチ開始処理
        _class.prototype.startTouch = function () {
            this._touching = true;
            this._touchMoving = false;
            this._touchCount = 0;
        };

        // ● タッチ処理
        _class.prototype.onTouch = function (triggered) {
            var lastSelector = this.selector();
            var x = this.canvasToLocalX(TouchInput.x);
            var y = this.canvasToLocalY(TouchInput.y);
            var hitSelector = this.hitTest(x, y);
            if (!triggered && lastSelector !== hitSelector) {
                this._touchMoving = true;
            }
            if (this.isCursorMovable() && this.isValidSelector(hitSelector)) {
                this.select(hitSelector);
            }
            if (!triggered && lastSelector !== this.selector()) {
                this.playTouchSound();
            }
            if (!triggered) {
                this._touchCount++;
            }
        };

        // ● タッチ終了処理
        _class.prototype.endTouch = function () {
            if (this.isTouchOkEnabled() && !this._touchMoving && this._touchCount < this.touchOkFrame() && this.isSelected()) {
                this.processOk();
            }
        };

        // ● タッチしていない処理
        _class.prototype.nonTouch = function () {
            this._touching = false;
            this._touchMoving = false;
            this._touchCount = 0;
        };

        // ● タッチ決定フレーム数の取得
        _class.prototype.touchOkFrame = function () {
            return 12;
        };

    };

    //=========================================================================
    // EXPerch_CommandSprites
    // 　画像選択に、コマンド画像の機能を追加します。
    //=========================================================================

    $.EXPerch_CommandSprites = function (_class, _super) {

        // メモ : initialize で画像のクリア、 refresh で画像のリサイズを行う。

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● コマンド画像の生成 (継承先で再定義する)
        _class.prototype.generateCommandSprite = function () {
            var layer = this.contentsLayer();
            return new $.Sprite_Drawing(0, 0, layer.width, layer.height);
        };

        // ● コマンド画像のクリア
        _class.prototype.clearCommandSprites = function () {
            this._commandSprites = [];
        };

        // ● コマンド画像のリサイズ
        _class.prototype.resizeCommandSprites = function (size) {
            var layer = this.contentsLayer();
            while (this._commandSprites.length < size) {
                var sprite = this.generateCommandSprite();
                this._commandSprites.push(sprite);
                layer.addChild(sprite);
            }
            while (this._commandSprites.length > size) {
                layer.removeChild(this._commandSprites.pop());
            }
        };

        //---------------------------------------------------------------------
        // 項目処理
        //---------------------------------------------------------------------

        // ● 選択可能なオブジェクト配列を取得
        _class.prototype.selectableObjects = function () {
            return this._commandSprites;
        };

    };

    //=========================================================================
    // EXPerch_CommandList
    // 　画像選択に、コマンドリスト管理機能を追加します。
    //=========================================================================

    $.EXPerch_CommandList = function (_class, _super) {

        // メモ : initialize と refresh でリストのクリアと作成を行う。

        //---------------------------------------------------------------------
        // 項目処理
        //---------------------------------------------------------------------

        // ● 要素数の取得
        _class.prototype.maxItems = function () {
            return this._list.length;
        };

        //---------------------------------------------------------------------
        // コマンド処理
        //---------------------------------------------------------------------

        // メモ : コマンドリストの作成 makeCommandList は継承先で定義する。

        // ● コマンドリストのクリア
        _class.prototype.clearCommandList = function () {
            this._list = [];
        };

        // ● コマンドの追加
        _class.prototype.addCommand = function (name, symbol, enabled, ext) {
            if (enabled === undefined) {
                enabled = true;
            }
            if (ext === undefined) {
                ext = null;
            }
            this._list.push({
                name: name,
                symbol: symbol,
                enabled: enabled,
                ext: ext
            });
        };

        // ● コマンド名の取得
        _class.prototype.commandName = function (selector) {
            return this._list[selector].name;
        };

        // ● シンボルの取得
        _class.prototype.commandSymbol = function (selector) {
            return this._list[selector].symbol;
        };

        // ● 選択可能状態の取得
        _class.prototype.isCommandEnabled = function (selector) {
            return this._list[selector].enabled;
        };

        // ● 現在の項目の取得
        _class.prototype.currentData = function () {
            return this.selector() >= 0 ? this._list[this.selector()] : null;
        };

        // ● 現在のシンボルの取得
        _class.prototype.currentSymbol = function () {
            return this.currentData() ? this.currentData().symbol : null;
        };

        // ● 現在の拡張情報の取得
        _class.prototype.currentExt = function () {
            return this.currentData() ? this.currentData().ext : null;
        };

        // ● シンボルからインデックスを取得
        _class.prototype.findSymbol = function (symbol) {
            for (var i = 0; i < this._list.length; i++) {
                if (this._list[i].symbol === symbol) {
                    return i;
                }
            }
            return -1;
        };

        // ● シンボルから項目を選択
        _class.prototype.selectSymbol = function (symbol) {
            var selector = this.findSymbol(symbol);
            if (selector >= 0) {
                this.select(selector);
            } else {
                this.select(0);
            }
        };

        // ● 拡張情報からインデックスを取得
        _class.prototype.findExt = function (ext) {
            for (var i = 0; i < this._list.length; i++) {
                if (this._list[i].ext === ext) {
                    return i;
                }
            }
            return -1;
        };

        // ● 拡張情報から項目を選択
        _class.prototype.selectExt = function (ext) {
            var selector = this.findExt(ext);
            if (selector >= 0) {
                this.select(selector);
            } else {
                this.select(0);
            }
        };

        //---------------------------------------------------------------------
        // ハンドラ処理
        //---------------------------------------------------------------------

        // ● 決定可能判定
        _class.prototype.isOkEnabled = function () {
            return true;
        };

        // ● 現在の項目の選択可能判定
        _class.prototype.isCurrentItemEnabled = function () {
            return this.currentData() ? this.currentData().enabled : false;
        };

        // ● 決定処理を呼び出す
        _class.prototype.callOkHandler = function () {
            var symbol = this.currentSymbol();
            if (this.isHandled(symbol)) {
                this.callHandler(symbol);
            } else if (this.isHandled('ok')) {
                _super.prototype.callOkHandler.call(this);
            } else {
                this.activate();
            }
        };

        //---------------------------------------------------------------------
        // サウンド処理
        //---------------------------------------------------------------------

        // ● 決定音を鳴らす
        _class.prototype.playOkSound = function () {
            if (this.currentSymbol() === 'cancel') {
                this.playCancelSound();
            } else {
                _super.prototype.playOkSound.call(this);
            }
        };

    };

    //=========================================================================
    // EXPerch_Compatible
    // 　画像選択に、一部ウィンドウ互換機能を追加します。
    //=========================================================================

    $.EXPerch_Compatible = function (_class, _super) {

        // メモ : ダミー画像の作成を行い、項目の描画とクリアを再定義する。
        
        // ● コンテンツ
        Object.defineProperty(_class.prototype, 'contents', {
            get: function () {
                return this._dummySprite.bitmap;
            },
            set: function (value) {
                this._dummySprite.bitmap = value;
            },
            configurable: true
        });

        // ● ダミー画像の作成
        _class.prototype.createDummySprite = function () {
            this._dummySprite = new $.Sprite_Drawing(0, 0, 0, 0);
            this.addChild(this._dummySprite);
        };
        
        // ● インデックスの取得
        _class.prototype.index = function () {
            return this.selector();
        };

        // ● 項目領域の取得
        _class.prototype.itemRect = function (index) {
            return this.targetRect(this.selectableObjects()[index]);
        };

        // ● 項目の描画 (継承先で再定義する)
        _class.prototype.drawItem = function (index) {};

        // ● 項目のクリア (継承先で再定義する)
        _class.prototype.clearItem = function (index) {};

        // ● 項目の再描画
        _class.prototype.redrawItem = function (index) {
            if (index >= 0) {
                this.clearItem(index);
                this.drawItem(index);
            }
        };

        // ● 現在の項目の再描画
        _class.prototype.redrawCurrentItem = function () {
            this.redrawItem(this.index());
        };
    };

    //=========================================================================
    // Perch_Scroll
    // 　画像をスクロール選択できるコマンドのパーチクラスです。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Perch_Scroll() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.Perch_Selection;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Perch_Scroll = _class;

        // ● 機能拡張
        $.EXPerch_DistanceSelection(_class, _super);

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function (x, y, width, height) {
            _super.prototype.initialize.call(this, x, y, width, height);
            this._scrollX = 0;
            this._scrollY = 0;
            this._cursorCount = 0;
            this._cursorMoveX = 0;
            this._cursorMoveY = 0;
        };

        // ● フレーム更新
        _class.prototype.update = function () {
            _super.prototype.update.call(this);
            this.updateCursorState();
            this.updatePosition();
        };

        // ● 位置の更新
        _class.prototype.updatePosition = function () {
            this.updateCommands();
            _super.prototype.updatePosition.call(this);
        };

        //---------------------------------------------------------------------
        // カーソル処理
        //---------------------------------------------------------------------

        // ● カーソル移動可能判定
        _class.prototype.isCursorMovable = function () {
            return (_super.prototype.isCursorMovable.call(this) && this._cursorCount <= 0);
        };

        // ● カーソルが見える位置に移動
        _class.prototype.ensureCursorVisible = function () {
            if (this.isSelected()) {
                this.clearCursorState();
                var x = this.placementX(this.selector());
                var y = this.placementY(this.selector());
                var maxWidth = this.maxWidth() - this.itemWidth();
                var maxHeight = this.maxHeight() - this.itemHeight();
                var limitX = Math.max(Math.min(x, maxWidth), 0);
                var limitY = Math.max(Math.min(y, maxHeight), 0);
                this.moveScrollX(this._scrollX + x - limitX);
                this.moveScrollY(this._scrollY + y - limitY);
                if (x !== limitX || y !== limitY) {
                    this._cursorMoveX = x - limitX;
                    this._cursorMoveY = y - limitY;
                    this._cursorCount = this.scrollFrame();
                }
            }
        };

        // ● カーソル状態のクリア
        _class.prototype.clearCursorState = function () {
            this._cursorCount = 0;
            this._cursorMoveX = 0;
            this._cursorMoveY = 0;
        };

        // ● カーソルカウンタの更新
        _class.prototype.updateCursorState = function () {
            if (this._cursorCount > 0) {
                this._cursorCount = Math.max(this._cursorCount - 1, 0);
            }
        };

        //---------------------------------------------------------------------
        // タッチ処理
        //---------------------------------------------------------------------

        // ● タッチ操作可能判定
        _class.prototype.isTouchable = function () {
            return this.isOpenAndActive() && this._cursorCount <= 0;
        };

        // ● タッチ開始処理
        _class.prototype.startTouch = function () {
            this._touching = true;
            this._touchMoving = false;
            this._touchStartX = this.canvasToLocalX(TouchInput.x);
            this._touchStartY = this.canvasToLocalY(TouchInput.y);
            this._touchPreviousX = this._touchStartX;
            this._touchPreviousY = this._touchStartY;
            this._touchCurrentX = this._touchStartX;
            this._touchCurrentY = this._touchStartY;
        };

        // ● タッチ処理
        _class.prototype.onTouch = function (triggered) {
            this._touchPreviousX = this._touchCurrentX;
            this._touchPreviousY = this._touchCurrentY;
            this._touchCurrentX = this.canvasToLocalX(TouchInput.x);
            this._touchCurrentY = this.canvasToLocalY(TouchInput.y);
            if (!this._touchMoving && this.checkTouchMovement()) {
                this._touchMoving = true;
            }
            if (this._touchMoving) {
                var dx = this._touchCurrentX - this._touchPreviousX;
                var dy = this._touchCurrentY - this._touchPreviousY;
                this.moveScrollX(this._scrollX - dx);
                this.moveScrollY(this._scrollY - dy);
            }
        };

        // ● タッチ終了処理
        _class.prototype.endTouch = function () {
            if (!this._touchMoving) {
                var lastSelector = this.selector();
                var x = this.canvasToLocalX(TouchInput.x);
                var y = this.canvasToLocalY(TouchInput.y);
                var hitSelector = this.hitTest(x, y);
                if (lastSelector === hitSelector && this.isSelected()) {
                    if (this.isTouchOkEnabled()) {
                        this.processOk();
                    }
                } else if (this.isCursorMovable() && this.isValidSelector(hitSelector)) {
                    this.select(hitSelector);
                    if (lastSelector !== this.selector()) {
                        this.playTouchSound();
                    }
                }
            }
        };

        // ● タッチしていない処理
        _class.prototype.nonTouch = function () {
            this._touching = false;
            this._touchMoving = false;
            this._touchStartX = 0;
            this._touchStartY = 0;
            this._touchPreviousX = 0;
            this._touchPreviousY = 0;
            this._touchCurrentX = 0;
            this._touchCurrentY = 0;
        };

        // ● タッチ移動判定
        _class.prototype.checkTouchMovement = function () {
            var mx = this.canvasToLocalX(TouchInput.x) - this._touchStartX;
            var my = this.canvasToLocalY(TouchInput.y) - this._touchStartY;
            var d = this.touchMovementThreshold();
            return (Math.abs(mx) > d || Math.abs(my) > d);
        };

        // ● 移動判定距離の取得
        _class.prototype.touchMovementThreshold = function () {
            return 24;
        };

        //---------------------------------------------------------------------
        // 配置処理
        //---------------------------------------------------------------------

        // ● 項目の幅の取得 (継承先で再定義する)
        _class.prototype.itemWidth = function () {
            return this.maxWidth();
        };

        // ● 項目の高さの取得 (継承先で再定義する)
        _class.prototype.itemHeight = function () {
            return 80;
        };

        // ● コマンド列数の取得 (継承先で再定義する)
        _class.prototype.maxCols = function () {
            return 1;
        };

        // ● コマンド行数の取得
        _class.prototype.maxRows = function () {
            return Math.max(Math.ceil(this.maxItems() / this.maxCols()), 1);
        };

        // ● 最大幅の取得
        _class.prototype.maxWidth = function () {
            return this.contentsLayer().width;
        };

        // ● 最大高さの取得
        _class.prototype.maxHeight = function () {
            return this.contentsLayer().height;
        };

        // ● スクロール時間の取得
        _class.prototype.scrollFrame = function () {
            return 8;
        };

        // ● カーソル移動X座標の取得
        _class.prototype.cursorMovementX = function () {
            return this._cursorMoveX * this._cursorCount / this.scrollFrame();
        };

        // ● カーソル移動Y座標の取得
        _class.prototype.cursorMovementY = function () {
            return this._cursorMoveY * this._cursorCount / this.scrollFrame();
        };

        // ● 配置X座標の取得
        _class.prototype.placementX = function (selector) {
            var x = (selector % this.maxCols()) * this.itemWidth();
            x += this.cursorMovementX();
            x -= this._scrollX;
            return x;
        };

        // ● 配置Y座標の取得
        _class.prototype.placementY = function (selector) {
            var y = Math.floor(selector / this.maxCols()) * this.itemHeight();
            y += this.cursorMovementY();
            y -= this._scrollY;
            return y;
        };

        // ● X座標のスクロール
        _class.prototype.moveScrollX = function (x) {
            var span = this.itemWidth() * this.maxCols();
            var max = this.maxWidth();
            this._scrollX = Math.max(Math.min(x, span - max), 0);
        };

        // ● Y座標のスクロール
        _class.prototype.moveScrollY = function (y) {
            var span = this.itemHeight() * this.maxRows();
            var max = this.maxHeight();
            this._scrollY = Math.max(Math.min(y, span - max), 0);
        };

        // ● コマンド配置の更新
        _class.prototype.updateCommands = function () {
            if (this.visible) {
                var objects = this.selectableObjects();
                for (var i = 0; i < objects.length; i++) {
                    this.placingCommand(objects[i], i);
                }
            }
        };

        // ● コマンドの配置
        _class.prototype.placingCommand = function (object, selector) {
            var x = this.placementX(selector);
            var y = this.placementY(selector);
            object.x = x + (this.itemWidth() - object.width) / 2;
            object.y = y + (this.itemHeight() - object.height) / 2;
        };

    })();

    //=========================================================================
    // Perch_SlideRing
    // 　画像をスライド選択できるリング状コマンドのパーチクラスです。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Perch_SlideRing() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.Perch_Selection;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Perch_SlideRing = _class;

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function (x, y, width, height) {
            _super.prototype.initialize.call(this, x, y, width, height);
            this._angle = 0;
            this._cursorCount = 0;
            this._cursorDirection = 0;
            this._touchPreviousAngle = 0;
            this._touchCurrentAngle = 0;
        };

        // ● フレーム更新
        _class.prototype.update = function () {
            _super.prototype.update.call(this);
            this.updateCursorState();
            this.updatePosition();
        };

        // ● 位置の更新
        _class.prototype.updatePosition = function () {
            this.updateCommands();
            _super.prototype.updatePosition.call(this);
        };

        //---------------------------------------------------------------------
        // カーソル処理
        //---------------------------------------------------------------------

        // ● カーソル移動可能判定
        _class.prototype.isCursorMovable = function () {
            return (_super.prototype.isCursorMovable.call(this) && this._cursorCount <= 0);
        };

        // ● カーソルを右に移動
        _class.prototype.cursorRight = function (wrap) {
            this.select(Math.min(this.selector() + 1, this.maxItems() - 1));
            this.nonTouch();
        };

        // ● カーソルを左に移動
        _class.prototype.cursorLeft = function (wrap) {
            this.select(Math.max(this.selector() - 1, 0));
            this.nonTouch();
        };

        // ● カーソルが見える位置に移動
        _class.prototype.ensureCursorVisible = function () {
            if (this.isSelected()) {
                this.clearCursorState();
                var angle = this.placementAngle(this.selector());
                var min = this.minAngle();
                var max = this.maxAngle();
                var speed = this.rotationalSpeed();
                var limit = Math.max(Math.min(angle, max), min);
                this.moveAngle(this._angle + angle - limit);
                if (angle < min) {
                    this._cursorCount = (min - angle) / speed;
                    this._cursorDirection = -1;
                } else if (angle > max) {
                    this._cursorCount = (angle - max) / speed;
                    this._cursorDirection = 1;
                }
            }
        };

        // ● カーソル状態のクリア
        _class.prototype.clearCursorState = function () {
            this._cursorCount = 0;
            this._cursorDirection = 0;
        };

        // ● カーソル状態の更新
        _class.prototype.updateCursorState = function () {
            if (this._cursorCount > 0) {
                this._cursorCount = Math.max(this._cursorCount - 1, 0);
            }
        };

        //---------------------------------------------------------------------
        // タッチ処理
        //---------------------------------------------------------------------

        // ● タッチ操作可能判定
        _class.prototype.isTouchable = function () {
            return this.isOpenAndActive() && this._cursorCount <= 0;
        };

        // ● タッチ開始処理
        _class.prototype.startTouch = function () {
            var angle = this.coordinateAngle();
            this._touching = true;
            this._touchMoving = false;
            this._touchStartX = this.canvasToLocalX(TouchInput.x);
            this._touchStartY = this.canvasToLocalY(TouchInput.y);
            this._touchPreviousAngle = angle;
            this._touchCurrentAngle = angle;
        };

        // ● タッチ処理
        _class.prototype.onTouch = function (triggered) {
            this._touchPreviousAngle = this._touchCurrentAngle;
            this._touchCurrentAngle = this.coordinateAngle();
            if (!this._touchMoving && this.checkTouchMovement()) {
                this._touchMoving = true;
            }
            if (this._touchMoving) {
                var pre = this._touchPreviousAngle;
                var now = this._touchCurrentAngle;
                var angle = this.angularDifference(pre, now);
                this.moveAngle(this._angle - angle);
            }
        };

        // ● タッチ終了処理
        _class.prototype.endTouch = function () {
            if (!this._touchMoving) {
                var lastSelector = this.selector();
                var x = this.canvasToLocalX(TouchInput.x);
                var y = this.canvasToLocalY(TouchInput.y);
                var hitSelector = this.hitTest(x, y);
                if (lastSelector === hitSelector && this.isSelected()) {
                    if (this.isTouchOkEnabled()) {
                        this.processOk();
                    }
                } else if (this.isCursorMovable() && this.isValidSelector(hitSelector)) {
                    this.select(hitSelector);
                    if (lastSelector !== this.selector()) {
                        this.playTouchSound();
                    }
                }
            }
        };

        // ● タッチしていない処理
        _class.prototype.nonTouch = function () {
            this._touching = false;
            this._touchMoving = false;
            this._touchStartX = 0;
            this._touchStartY = 0;
            this._touchPreviousAngle = 0;
            this._touchCurrentAngle = 0;
        };

        // ● 座標から角度を取得
        _class.prototype.coordinateAngle = function () {
            var x = this.canvasToLocalX(TouchInput.x);
            var y = this.canvasToLocalY(TouchInput.y);
            var diffX = this.baseX() - x;
            var diffY = this.baseY() - y;
            return Math.atan2(-diffX, diffY);
        };

        // ● タッチ移動判定
        _class.prototype.checkTouchMovement = function () {
            var mx = this.canvasToLocalX(TouchInput.x) - this._touchStartX;
            var my = this.canvasToLocalY(TouchInput.y) - this._touchStartY;
            var d = this.touchMovementThreshold();
            return (Math.abs(mx) > d || Math.abs(my) > d);
        };

        // ● 移動判定距離の取得
        _class.prototype.touchMovementThreshold = function () {
            return 24;
        };

        // ● 二点間の角度差の取得
        _class.prototype.angularDifference = function (a1, a2) {
            if (a1 >= 0) {
                if (a2 < a1 - Math.PI) {
                    a2 += Math.PI * 2;
                }
            } else {
                if (a2 > a1 + Math.PI) {
                    a2 -= Math.PI * 2;
                }
            }
            return a2 - a1;
        };

        //---------------------------------------------------------------------
        // 配置処理
        //---------------------------------------------------------------------

        // ● 基準のX座標の取得 (継承先で再定義する)
        _class.prototype.baseX = function () {
            return 0;
        };

        // ● 基準のY座標の取得 (継承先で再定義する)
        _class.prototype.baseY = function () {
            return 0;
        };

        // ● 基準の半径の取得 (継承先で再定義する)
        _class.prototype.baseRadius = function () {
            return 0;
        };

        // ● ずらし角度の取得 (継承先で再定義する)
        _class.prototype.shiftAngle = function () {
            return 0;
        };

        // ● 最小角度の取得
        _class.prototype.minAngle = function () {
            return 0;
        };

        // ● 最大角度の取得
        _class.prototype.maxAngle = function () {
            return 0;
        };

        // ● 回転速度の取得
        _class.prototype.rotationalSpeed = function () {
            return Math.PI * 0.03;
        };

        // ● カーソル移動角度の取得
        _class.prototype.cursorMovementAngle = function () {
            return this._cursorDirection * this._cursorCount * this.rotationalSpeed();
        };

        // ● 配置角度の取得
        _class.prototype.placementAngle = function (selector) {
            var angle = this.shiftAngle() * selector;
            angle += this.cursorMovementAngle();
            angle -= this._angle;
            return angle;
        };

        // ● 角度の移動
        _class.prototype.moveAngle = function (angle) {
            var span = this.shiftAngle() * (this.maxItems() - 1);
            var max = this.maxAngle();
            var min = this.minAngle();
            this._angle = Math.max(Math.min(angle, span - max), min);
        };

        // ● コマンド配置の更新
        _class.prototype.updateCommands = function () {
            if (this.visible) {
                var objects = this.selectableObjects();
                for (var i = 0; i < objects.length; i++) {
                    this.placingCommand(objects[i], i);
                }
            }
        };

        // ● コマンドの配置
        _class.prototype.placingCommand = function (object, selector) {
            var x = this.baseX();
            var y = this.baseY();
            if (this.openness > 0) {
                var angle = this.placementAngle(selector);
                var radius = this.baseRadius() * this.openness / 255;
                x += radius * Math.sin(angle);
                y -= radius * Math.cos(angle);
            }
            object.x = x - object.width / 2;
            object.y = y - object.height / 2;
        };

    })();

    //=========================================================================
    // Perch_Command
    // 　コマンドリスト方式の画像コマンド選択のベースとなるクラスです。
    //=========================================================================

    (function () {

        //---------------------------------------------------------------------
        // クラス処理
        //---------------------------------------------------------------------

        // ● クラスの生成
        var _class = function Perch_Command() {
            this.initialize.apply(this, arguments);
        };

        // ● スーパークラス
        var _super = $.Perch_Selection;

        // ● クラスの継承
        _class.prototype = Object.create(_super.prototype);
        _class.prototype.constructor = _class;

        // ● 関数化
        $.Perch_Command = _class;

        // ● 機能拡張
        $.EXPerch_DistanceSelection(_class, _super);
        $.EXPerch_TouchSelection(_class, _super);
        $.EXPerch_CommandSprites(_class, _super);
        $.EXPerch_CommandList(_class, _super);

        //---------------------------------------------------------------------
        // 基本処理
        //---------------------------------------------------------------------

        // ● 初期化
        _class.prototype.initialize = function (x, y, width, height) {
            _super.prototype.initialize.call(this, x, y, width, height);
            this.createSprites();
            this.refresh();
            this.select(0);
            this.activate();
        };

        // ● リフレッシュ
        _class.prototype.refresh = function () {
            this.clearCommandList();
            this.makeCommandList();
            this.refreshSprites();
        };

        //---------------------------------------------------------------------
        // 画像処理
        //---------------------------------------------------------------------

        // ● コマンド画像の生成
        _class.prototype.generateCommandSprite = function () {
            var width = this.commandWidth();
            var height = this.commandHeight();
            return new $.Sprite_Command(0, 0, width, height);
        };

        // ● 画像の作成
        _class.prototype.createSprites = function () {
            this.clearCommandSprites();
        };

        // ● 画像のリフレッシュ
        _class.prototype.refreshSprites = function () {
            this.resizeCommandSprites(this.maxItems());
            for (var i = 0; i < this.maxItems(); i++) {
                this.refreshCommandSprite(this._commandSprites[i], i);
            }
        };

        // ● コマンド画像のリフレッシュ
        _class.prototype.refreshCommandSprite = function (sprite, index) {
            var data = this._list[index];
            sprite.setList(data);
            this.placingCommand(sprite, index);
        };

        //---------------------------------------------------------------------
        // 配置処理
        //---------------------------------------------------------------------

        // ● コマンド幅の取得
        _class.prototype.commandWidth = function () {
            return 192;
        };

        // ● コマンド高さの取得
        _class.prototype.commandHeight = function () {
            return 64;
        };

        // ● 列数の取得
        _class.prototype.maxCols = function () {
            return 1;
        };

        // ● 行数の取得
        _class.prototype.maxRows = function () {
            return Math.ceil(this.maxItems() / this.maxCols());
        };

        // ● スペースの取得
        _class.prototype.spacing = function () {
            return 8;
        };

        // ● コマンドの配置
        _class.prototype.placingCommand = function (sprite, index) {
            var layer = this.contentsLayer();
            var s = this.spacing();
            var cols = this.maxCols();
            var rows = this.maxRows();
            var cw = (layer.width - s * 2) / cols;
            var ch = (layer.height - s * 2) / rows;
            var x = (index % cols) * cw + (cw - sprite.width) / 2 + s;
            var y = Math.floor(index / cols) * ch + (ch - sprite.height) / 2 + s;
            sprite.move(x, y);
        };

    })();

    //=========================================================================
    // Scene_Base
    //=========================================================================

    (function (_class, _super) {

        // ● パーチを追加
        _class.prototype.addPerch = function (child) {
            var index = this.children.indexOf(this._windowLayer);
            return this.addChildAt(child, index);
        };

    })(Scene_Base, Stage);

    //=========================================================================
    // Scene_Boot
    //=========================================================================

    (function (_class, _super) {

        // ● システム画像のロード
        var _loadSystemImages = _class.loadSystemImages;
        _class.loadSystemImages = function () {
            _loadSystemImages.call(this);
            ImageManager.reserveSystem(_imgPerchCursor);
            ImageManager.reserveSystem(_imgCommandSkin);
            ImageManager.reserveSystem(_imgIconRing);
        };

    })(Scene_Boot, Scene_Base);

})(Vitsuno);
